/*
 * Copyright (C) 2015 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.driver4j.bdrp.node.pool;

import org.apache.commons.pool2.PooledObject;
import org.apache.commons.pool2.PooledObjectFactory;
import org.apache.commons.pool2.impl.DefaultPooledObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.driver4j.bdrp.node.BdrpUtils;

import redis.clients.jedis.BinaryJedis;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.Protocol;

/**
 * BDRP Jedis生成工厂，参考JedisFactory，主要修改了validateObject方法
 *
 * @author dingxuefeng
 */
public class BdrpFactory implements PooledObjectFactory<Jedis> {
    private static final Logger logger = LoggerFactory.getLogger(BdrpFactory.class);

    private final String host;
    private final int port;
    private final int timeout;
    private final String password;
    private final int database;
    private final String clientName;

    /**
     * 初始化工厂
     *
     * @param host     主机
     * @param port     端口
     * @param timeout  超时
     * @param password 密码
     * @param database 数据库
     */
    public BdrpFactory(final String host, final int port, final int timeout,
                       final String password, final int database) {
        this(host, port, timeout, password, database, null);
    }

    /**
     * 初始化工厂
     *
     * @param host       主机
     * @param port       端口
     * @param timeout    超时
     * @param password   密码
     * @param database   数据库
     * @param clientName 客户端名称
     */
    public BdrpFactory(final String host, final int port, final int timeout,
                       final String password, final int database, final String clientName) {
        super();
        this.host = host;
        this.port = port;
        this.timeout = timeout;
        this.password = password;
        this.database = database;
        this.clientName = clientName;
    }

    /**
     * 激活对象
     *
     * @param pooledJedis 池中的Jedis对象
     * @throws Exception 激活时可能抛出异常
     */
    @Override
    public void activateObject(PooledObject<Jedis> pooledJedis) throws Exception {
        BinaryJedis jedis = pooledJedis.getObject();
        if (jedis.getDB() != database) {
            jedis.select(database);
        }
    }

    /**
     * 销毁对象
     *
     * @param pooledJedis 需要销毁的Jedis
     * @throws Exception 应该不会抛出异常的
     */
    @Override
    public void destroyObject(PooledObject<Jedis> pooledJedis) throws Exception {
        BinaryJedis jedis = pooledJedis.getObject();
        if (!jedis.isConnected()) {
            return;
        }
        try {
            try {
                jedis.quit();
            } catch (Exception e) {
            }
            jedis.disconnect();
        } catch (Exception e) {
        }
    }

    /**
     * 构造连接池中的Jedis对象
     *
     * @return 连接好的Jedis
     * @throws Exception 构造对象可能会抛出异常
     */
    @Override
    public PooledObject<Jedis> makeObject() throws Exception {
        final Jedis jedis = new Jedis(this.host, this.port, this.timeout);

        jedis.connect();
        if (null != this.password) {
            jedis.auth(this.password);
        }
        if (database != Protocol.DEFAULT_DATABASE) {
            jedis.select(database);
        }
        if (clientName != null) {
            jedis.clientSetname(clientName);
        }

        return new DefaultPooledObject<Jedis>(jedis);
    }

    /**
     * 暂时啥事都不用干
     *
     * @param pooledJedis Jedis对象
     * @throws Exception 暂时不会抛出异常
     */
    @Override
    public void passivateObject(PooledObject<Jedis> pooledJedis)
            throws Exception {
        // TODO maybe should select db 0? Not sure right now.
    }

    /**
     * 验证Jedis连接，这里判定是否已连接，随便取个KEY，看看会不会抛异常，不抛异常就OK
     *
     * @param pooledJedis 给定的Jedis
     * @return 验证结果
     */
    @Override
    public boolean validateObject(PooledObject<Jedis> pooledJedis) {
        return BdrpUtils.validateJedis(pooledJedis.getObject());
    }


}
