package com.baidu.bigpipe.driver;

import java.util.concurrent.atomic.AtomicBoolean;

import com.baidu.bigpipe.driver.converter.pub.JsonMessageConverter;
import com.baidu.bigpipe.driver.converter.pub.MessageConverter;
import com.baidu.bigpipe.driver.converter.pub.StringMessageConverter;
import com.baidu.bigpipe.protocol.BigpipePackBuilder;
import com.baidu.bigpipe.protocol.DefaultBigpipePackBuilder;
import com.baidu.bigpipe.protocol.LogIdGen;
import com.baidu.bigpipe.protocol.SequenceLogIdGen;
import com.baidu.bigpipe.transport.conf.BigPipeConf;
import com.baidu.bigpipe.transport.conf.SocketConf;
import com.baidu.bigpipe.transport.pub.AsynchronousPublisher;
import com.baidu.bigpipe.transport.pub.AsynchronousPublisherImpl;
import com.baidu.bigpipe.transport.pub.PublishStrategy;

/**
 * bigpipe 异步发送消息客户端的抽象实现，它实现一些公用的方法。
 * 
 * @author hexiufeng
 * 
 */
public abstract class AbstractBigpipeClient {
    /**
     * 需要先转化为json格式 消息转换器
     */
    protected MessageConverter converter = new JsonMessageConverter();
    /**
     * String类型的消息的转化器
     */
    protected MessageConverter stringConverter = new StringMessageConverter();
    /**
     * 发布策略
     */
    protected PublishStrategy pubStrategy;
    protected LogIdGen logIdGen = new SequenceLogIdGen();
    protected AsynchronousPublisher publisher;
    protected BigPipeConf conf;
    protected SocketConf socketConf = new SocketConf();
    
    private static final AtomicBoolean intialized = new AtomicBoolean(false);

    public MessageConverter getConverter() {
        return converter;
    }

    public void setConverter(MessageConverter converter) {
        this.converter = converter;
    }

    public BigPipeConf getConf() {
        return conf;
    }

    public void setConf(BigPipeConf conf) {
        this.conf = conf;
    }

    public LogIdGen getLogIdGen() {
        return logIdGen;
    }

    public void setLogIdGen(LogIdGen logIdGen) {
        this.logIdGen = logIdGen;
    }

    /**
     * 构建出支持本客户端规范的{@link com.baidu.bigpipe.protocol.BigpipePackBuilder BigpipePackBuilder}实例。 之所以使用匿名类是想公用被客户端的消息转化规范。
     * 
     * @return {@link BigpipePackBuilder}用于打包的对象
     */
    public final BigpipePackBuilder newBigpipePackBuilder() {
        return new DefaultBigpipePackBuilder();
    }

    /**
     * 初始化,通常在spring的init-method中配置
     */
    public void init() {
        if (conf == null) {
            throw new RuntimeException("you should configure bigpipe info.");
        }
        
        if (intialized.compareAndSet(false, true)) {
        	if (publisher == null) {
                AsynchronousPublisherImpl tmp = new AsynchronousPublisherImpl();
                tmp.setPubStrategy(createPublishStrategy());
                tmp.setSocketConf(socketConf);
                publisher = tmp;
            }
            publisher.applyMessageIdGen(logIdGen);
            publisher.init(conf);
        } else {
        	throw new RuntimeException("Client already called initialzed, do not call again.");
        }
    }

    /**
     * 关闭该客户端
     */
    public void shutDown() {
        if (publisher != null) {
            publisher.shutDown();
        }
        
        intialized.compareAndSet(true, false);
    }

    /**
     * 生成消息的发布策略,被模板方法调用
     * 
     * @return {@link PublishStrategy}发布策略
     */
    protected abstract PublishStrategy createPublishStrategy();
}
