package com.baidu.bigpipe.driver;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.bigpipe.driver.converter.pub.MessageConverter;
import com.baidu.bigpipe.driver.converter.pub.MessageConverterHelper;
import com.baidu.bigpipe.exp.PushException;
import com.baidu.bigpipe.protocol.BigpipePackBuilder;
import com.baidu.bigpipe.transport.pub.GroupPublishStrategy;
import com.baidu.bigpipe.transport.pub.Message;
import com.baidu.bigpipe.transport.pub.PublishStrategy;
import com.baidu.bigpipe.transport.pub.SendFutrue;

/**
 * 实现{@link com.baidu.bigpipe.driver.NoneBlockingBigpipePubClient
 * NoneBlockingBigpipePubClient}接口。 支持异步发送批量，单个或者一包消息。 <br>
 * <b>注意:</b><br>
 * <ul>
 * <li>在双工模式下批量消息之间是<b>无序</b>的，这样能够保证发送的高效率，<b>在非双工模式下，批量消息发送无助于效率提升</b>。</li>
 * <li>如果消息发送要保证<br>
 * 有序并且高效</b>，请<b>打包</b>发送消息</li>
 * </ul>
 * 
 * @author hexiufeng
 * 
 */
public class SimpleNoneBlockingPubClient extends AbstractBigpipeClient
		implements NoneBlockingBigpipePubClient {
	private static final Logger LOGGER = LoggerFactory
			.getLogger(SimpleNoneBlockingPubClient.class);

	@Override
	public SendFutrue publish(Object object, MessageConverter messageConverter)
			throws PushException {
		byte[] message = MessageConverterHelper.convert(object,
				messageConverter);
		Message msg = convert2Message(message, false);
		if (LOGGER.isDebugEnabled()) {
			LOGGER.debug("Do publish ");
		}
		
		return super.publisher.publish(msg);
	}

	@Override
	protected PublishStrategy createPublishStrategy() {
		if (super.pubStrategy == null
				|| !(super.pubStrategy instanceof GroupPublishStrategy)) {
			return new GroupPublishStrategy();
		}
		return super.pubStrategy;
	}

	@Override
	public SendFutrue publishPack(BigpipePackBuilder packBuilder)
			throws PushException {
		if (packBuilder == null) {
			throw new RuntimeException("param 'packBuilder' is null.");
		}

		byte[] message = packBuilder.toByteArray();
		Message msg = convert2Message(message, true);
		return super.publisher.publish(msg);
	}

	/**
	 * 把java对象消息封装成为发送者内部可识别的Message
	 * 
	 * @param m
	 *            java对象消息
	 * @return {@link Message}，内部消息描述
	 */
	private Message convert2Message(byte[] buf, boolean packed) {
		long logId = logIdGen.genId();
		Message om = new Message(logId + "", buf, buf);
		om.setPackedMessage(packed);
		return om;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.baidu.bigpipe.driver.NoneBlockingBigpipePubClient#publish(byte[])
	 */
	@Override
	public SendFutrue publish(byte[] message) throws PushException {
		Message msg = convert2Message(message, false);
		return super.publisher.publish(msg);
	}

}
