package com.baidu.bigpipe.protocol;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.LinkedList;
import java.util.List;

import com.baidu.bigpipe.driver.converter.pub.MessageConverter;
import com.baidu.bigpipe.driver.converter.pub.MessageConverterHelper;

/**
 * 批量消息打包构建对象,支持默认最大2M的消息打包.
 * 最大消息打包可通过 {@code  void setMaxBuffer(int maxBuffer) }来设置
 *
 * @author hexiufeng
 * @author xiemalin
 */
public class DefaultBigpipePackBuilder implements BigpipePackBuilder {
    private final static int INT_SIZE = 4;
    private final List<byte[]> packedList = new LinkedList<byte[]>();
    private static final int BUFF_LIMIT = 2096128; // 2m-1024
    private int maxBuffer = BUFF_LIMIT;
    private ByteBuffer bodyBuf;
    private int currentLength = 0;

    /**
     * setter method for property maxBuffer
     *
     * @param maxBuffer the maxBuffer to set
     */
    public void setMaxBuffer(int maxBuffer) {
        if (maxBuffer > 0) {
            this.maxBuffer = maxBuffer;
        }
    }

    /**
     * 构造方法
     */
    public DefaultBigpipePackBuilder() {
        bodyBuf = ByteBuffer.allocate(maxBuffer);
        bodyBuf.order(ByteOrder.LITTLE_ENDIAN);
        bodyBuf.position(INT_SIZE);
    }

    /**
     * 构造方法，支持指定buffer的大小
     *
     * @param buffSize buffer的大小，按byte作为单位
     */
    protected DefaultBigpipePackBuilder(int buffSize) {
        maxBuffer = buffSize;
        bodyBuf = ByteBuffer.allocate(maxBuffer);
        bodyBuf.order(ByteOrder.LITTLE_ENDIAN);
        bodyBuf.position(INT_SIZE);
    }

    /**
     * 返回true表示有可用的空间
     *
     * @return 返回4
     */
    private boolean checkRemaing(byte[] buf) {
        if (bodyBuf.remaining() >= buf.length + INT_SIZE) {
            return true;
        }
        return false;
    }

    @Override
    public boolean add(byte[] buf) {
        if (buf == null) {
            throw new RuntimeException("message is null");
        }
        if (checkRemaing(buf)) {
            bodyBuf.putInt(buf.length);
            bodyBuf.put(buf);
            packedList.add(buf);
            currentLength += buf.length + 4;
            return true;
        }
        return false;
    }

    @Override
    public void reset() {
        bodyBuf.position(INT_SIZE);
        bodyBuf.limit(maxBuffer);
        packedList.clear();
        currentLength = 0;
    }

    @Override
    public boolean hasMessages() {
        return packedList.isEmpty();
    }

    @Override
    public List<byte[]> getPackedMessage() {
        return packedList;
    }

    /**
     * 返回buffer的内容长度
     *
     * @return 长度
     */
    protected int getCurrentLen() {
        return bodyBuf.limit();
    }

    /* (non-Javadoc)
     * @see com.baidu.bigpipe.protocol.BigpipePackBuilder#add(com.baidu.bigpipe.driver.converter.pub.MessageConverter)
     */
    @Override
    public boolean add(Object object, MessageConverter messageConverter) {
        byte[] message = MessageConverterHelper.convert(object, messageConverter);
        return add(message);
    }

    /* (non-Javadoc)
     * @see com.baidu.bigpipe.protocol.BigpipePackBuilder#toByteArray()
     */
    @Override
    public byte[] toByteArray() {
        int totalLength = INT_SIZE + currentLength;
        byte[] pack = new byte[totalLength]; // 前四个字节记录后面body的长度,currentLength是body的长度
        bodyBuf.position(0);
        bodyBuf.putInt(currentLength);
        System.arraycopy(bodyBuf.array(), 0, pack, 0, totalLength);
        return pack;
    }

}
