package com.baidu.bigpipe.protocol.meta;

import java.net.InetSocketAddress;

import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.KeeperException.Code;

import com.baidu.bigpipe.common.support.JsonUtils;
import com.baidu.bigpipe.protocol.meta.concept.Broker;
import com.baidu.bigpipe.protocol.meta.concept.BrokerGroup;
import com.baidu.bigpipe.protocol.meta.concept.Pipelet;
import com.baidu.bigpipe.protocol.meta.concept.QueueAddress;
import com.baidu.bigpipe.protocol.meta.concept.Stripe;
import com.baidu.bigpipe.protocol.meta.concept.TopicAddress;
import com.baidu.bigpipe.protocol.meta.exp.InvalidParameter;
import com.baidu.bigpipe.protocol.meta.exp.NameResolveException;
import com.baidu.bigpipe.protocol.meta.exp.NameServiceUnkownException;
import com.baidu.bigpipe.protocol.meta.exp.QueueLocateException;
import com.baidu.bigpipe.protocol.meta.exp.StripeOffsetException;

/**
 * 名字服务，负责从bigpipe zk服务中心获取bigpipe地址信息
 *
 * @author hexiufeng
 */
public final class NameService {
    private PipeletInstanceRoleSelectStrategy roleStrategy = new DefaultPipeletInstanceRoleStrategy();
    private MetaLoader metaLoader;

    public MetaLoader getMetaLoader() {
        return metaLoader;
    }

    public void setMetaLoader(MetaLoader metaLoader) {
        this.metaLoader = metaLoader;
    }

    public void setClusterName(String clusterName) {
        this.clusterName = clusterName;
    }

    private String clusterName;

    public NameService(String topDomainName) {
        clusterName = topDomainName;
    }

    public String getClusterName() {
        return clusterName;
    }

    public PipeletInstanceRoleSelectStrategy getRoleStrategy() {
        return roleStrategy;
    }

    public void setRoleStrategy(PipeletInstanceRoleSelectStrategy roleStrategy) {
        this.roleStrategy = roleStrategy;
    }

    /**
     * 订阅时，从zk中根据pipelet名称获取具体实例的地址
     *
     * @param pipelet  pipelet名称
     * @param position 订阅点
     *
     * @return {@link TopicAddress} topic地址
     *
     * @throws NameResolveException 名字解析异常
     * @throws KeeperException      zk异常
     */
    public TopicAddress lookupForSub(String pipelet, long position) throws NameResolveException, KeeperException {
        return lookup(pipelet, position, roleStrategy.getCurrentRole());
    }

    /**
     * 发布时，从zk中根据pipelet名称获取具体实例的地址
     *
     * @param pipelet pipelet名称
     *
     * @return {@link TopicAddress} topic地址
     *
     * @throws NameResolveException 名字解析异常
     * @throws KeeperException      zk异常
     */
    public TopicAddress lookupForPub(String pipelet) throws NameResolveException, KeeperException {
        return lookup(pipelet, Long.MAX_VALUE, PipeletInstanceRoleSelectStrategy.MASTER_ROLE);
    }

    /**
     * 根据position或者pipelet的可用的stripe信息。 有关{@link Stripe} 参照bigpipe的概念订阅wiki
     *
     * @param pipelet  pipelet名称
     * @param position 订阅点位置
     *
     * @return {@link Stripe} Stripe信息，
     *
     * @throws NameResolveException 名字解析异常
     * @throws KeeperException      zk异常
     */
    private Stripe findStripe(String pipelet, long position) throws NameResolveException, KeeperException {
        String pipeletPath = "/" + clusterName + "/meta/pipelet/" + pipelet;
        String pipeletinfo;

        pipeletinfo = metaLoader.getMeta(pipeletPath);
        Pipelet p = (Pipelet) JsonUtils.json2Object(pipeletinfo, Pipelet.class);
        if (p.getStripes() != null && p.getStripes().size() > 0) {
            long minBeginPos = Long.MAX_VALUE;
            Stripe oldestStripe = null;
            for (Stripe s : p.getStripes()) {
                if (s.getBeginPos() > 0 && s.getBeginPos() <= s.getEndPos()) {
                    if (s.getBeginPos() < minBeginPos) {
                        minBeginPos = s.getBeginPos();
                        oldestStripe = s;
                    }
                    if (s.getBeginPos() <= position && s.getEndPos() >= position) {
                        return s;
                    }
                }
            }
            if (position == 0) {
                return oldestStripe;
            }
            throw new StripeOffsetException(pipelet, position, oldestStripe.getBeginPos());
        }
        throw new NameResolveException(pipelet, position,
                "stripe for offset " + position + " failed: no stripe exists");
    }

    /**
     * 根据Stripe信息获取{@link BrokerGroup}信息
     *
     * @param s {@link Stripe}对象
     *
     * @return {@link BrokerGroup}对象
     *
     * @throws KeeperException zk异常
     */
    private BrokerGroup getGroup(Stripe s) throws KeeperException {
        String groupPath = String.format("/%s/meta/broker_group/%s", clusterName, s.getServingGroup());
        String groupInfo = metaLoader.getMeta(groupPath);
        return (BrokerGroup) JsonUtils.json2Object(groupInfo, BrokerGroup.class);
    }

    /**
     * 从zk中获取pipelet的地址
     *
     * @param pipelet  pipelet名称
     * @param position 订阅点位置，for sub
     * @param role     获取的pipelet实例的角色，value of:
     *                 <ul>
     *                 <li>{@link PipeletInstanceRoleSelectStrategy#MASTER_ROLE}</li>
     *                 <li>{@link PipeletInstanceRoleSelectStrategy#SLAVE_ROLE}</li>
     *                 </ul>
     *
     * @return {@link TopicAddress} topic地址
     *
     * @throws NameResolveException 名字解析异常
     * @throws KeeperException      zk异常
     */
    private TopicAddress lookup(String pipelet, long position, int role)
            throws NameResolveException, KeeperException {
        Stripe stripe;
        try {
            stripe = findStripe(pipelet, position);
        } catch (KeeperException e) {
            if (e.code() == Code.NONODE || e.code() == Code.NOAUTH) {
                throw new InvalidParameter("no access to path " + e.getPath(), e);
            }
            throw e;
        }

        BrokerGroup group;
        try {
            group = getGroup(stripe);
        } catch (KeeperException e) {
            if (e.code() == Code.NONODE || e.code() == Code.NOAUTH) {
                throw new NameServiceUnkownException(
                        "zookeeper information inconsistency! no access to " + e.getPath(), e);
            }
            throw e;
        }

        TopicAddress addr = new TopicAddress();
        addr.setStripe(stripe);
        if (group.getBrokers() != null && group.getBrokers().size() > 0) {
            for (Broker broker : group.getBrokers()) {
                if (broker.getRole() == role) {
                    addr.setAddress(new InetSocketAddress(broker.getIp(), broker.getPort()));
                    return addr;
                }
            }
        }
        throw new NameServiceUnkownException("zookeeper information inconsistency! group " + stripe.getServingGroup()
                + " information is abnormal");
    }

    /**
     * 从zk中获取queue的地址
     *
     * @param queueName queueName
     *
     * @return {@link QueueAddress} queue地址
     *
     * @throws QueueLocateException 名字解析异常
     * @throws KeeperException      zk异常
     */
    public QueueAddress lookupQueue(String queueName)
            throws QueueLocateException, KeeperException {

        String queueAddr = "";
        try {
            String queuePath = String.format(
                    "/%s/meta/queue/%s", clusterName, queueName);
            if (metaLoader.stat(queuePath) == null) {
                throw new InvalidParameter(
                        "queue " + queueName + " not exist in "
                                + "cluster " + clusterName);
            }
            String queueNamePath = String.format(
                    "/%s/_register/%s", clusterName, queueName);
            byte[] bytes = metaLoader.get(queueNamePath);
            queueAddr = new String(bytes).replaceAll("\0", "");
        } catch (KeeperException e) {
            if (e.code() == Code.NONODE || e.code() == Code.NOAUTH) {
                QueueLocateException ex =
                        new QueueLocateException(queueName, "can't location queue, may be dead or not started", e);
                throw ex;
            }
        } catch (InterruptedException e) {
            QueueLocateException ex = new QueueLocateException(queueName, "reading zookeeper interrupted", e);
            throw ex;
        }
        String[] parts = queueAddr.split(":");
        try {
            InetSocketAddress intAddress = new InetSocketAddress(parts[0], Integer.parseInt(parts[1]));
            QueueAddress addr = new QueueAddress();
            addr.setAddress(intAddress);
            addr.setQueueName(queueName);
            return addr;
        } catch (Exception e) {
            QueueLocateException ex = new QueueLocateException(queueName, "process queue address failed", e);
            throw ex;
        }
    }
}
