package com.baidu.bigpipe.protocol.meta;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;

import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 从zookeeper中加载bigpipe的元数据的加载器实现
 *
 * @author hexiufeng
 */
public class ZKMetaLoader implements MetaLoader {
    private static final Logger LOGGER = LoggerFactory.getLogger(ZKMetaLoader.class);

    private String connectString;

    public String getConnectString() {
        return connectString;
    }

    public void setConnectString(String connectString) {
        this.connectString = connectString;
    }

    private ZooKeeper init() {
        ZooKeeper newZk = null;
        try {
            newZk = buildZookeeper();
        } catch (IOException e) {
            LOGGER.error("init zookeeper error", e);
            throw new RuntimeException(e);
        } catch (InterruptedException e) {
            LOGGER.error("init zookeeper error, ", e);
            throw new RuntimeException(e);
        }
        return newZk;
    }

    @Override
    public synchronized String getMeta(String path) throws KeeperException {
        ZooKeeper zk = init();
        Stat stat = new Stat();
        byte[] meta = null;
        try {
            meta = zk.getData(path, false, stat);
        } catch (InterruptedException e) {
            // ignore
        } finally {
            safeClose(zk);
        }
        return new String(meta, 8, meta.length - 8);
    }

    @Override
    public synchronized Stat stat(String path) throws KeeperException, InterruptedException {
        ZooKeeper zk = init();
        Stat stat = null;
        try {
            stat = zk.exists(path, false);
        } catch (InterruptedException e) {
            // ignore
        } finally {
            safeClose(zk);
        }
        return stat;
    }

    @Override
    public synchronized byte[] get(String path) throws KeeperException {
        ZooKeeper zk = init();
        Stat stat = new Stat();
        byte[] meta = null;
        try {
            meta = zk.getData(path, false, stat);
        } catch (InterruptedException e) {
            // ignore
        } finally {
            safeClose(zk);
        }
        return meta;
    }

    private synchronized void safeClose(ZooKeeper zk) {
        if (zk == null) {
            return;
        }
        try {
            zk.close();
        } catch (InterruptedException e) {
            // ignore
        } finally {
            zk = null;
        }
    }

    /**
     * 构建访问zookeeper的客户端对象
     *
     * @return {@link ZooKeeper}ZooKeeper客户端描述
     *
     * @throws IOException          网络io异常
     * @throws InterruptedException 线程被中断异常，这种情况基本不会发生
     */
    private ZooKeeper buildZookeeper() throws IOException, InterruptedException {
        final CountDownLatch connectLatch = new CountDownLatch(1);
        LOGGER.info("build zookeeper client.");
        ZooKeeper newZk = new ZooKeeper(connectString, 5000, new Watcher() {

            @Override
            public void process(WatchedEvent event) {
                if (KeeperState.SyncConnected == event.getState()) {
                    connectLatch.countDown();
                    return;
                }
            }
        });
        try {
            connectLatch.await();
        } catch (InterruptedException e) {
            throw e;
        }
        return newZk;

    }

}
