/*
 * Copyright 2015 the original author or authors.
 *
 * Licensed under the BAIDU
 */
package com.baidu.bigpipe.spring.annotation;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.PropertyValues;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.util.NumberUtils;
import org.springframework.util.StringUtils;

import com.baidu.bigpipe.driver.DefaultListenerSubscriber;
import com.baidu.bigpipe.driver.DefaultQueueListenerSubscriber;
import com.baidu.bigpipe.driver.ListenerSubscriber;
import com.baidu.bigpipe.driver.NoneBlockingBigpipePubClient;
import com.baidu.bigpipe.driver.QueueListenerSubscriber;
import com.baidu.bigpipe.driver.SimpleNoneBlockingPubClient;
import com.baidu.bigpipe.driver.converter.sub.MessageBodyConverter;
import com.baidu.bigpipe.driver.converter.sub.StringMessageBodyConverter;
import com.baidu.bigpipe.position.store.SubcribePositionStore;
import com.baidu.bigpipe.transport.conf.BigPipeConf;
import com.baidu.bigpipe.transport.sub.BigpipeMessageListener;
import com.baidu.bigpipe.transport.sub.PipeletIdAwareBigpipeMessageListener;

/**
 * Annotation resolver for {@link MessageListener} and {@link MessageSender}
 * 
 * @see CommonAnnotationBeanPostProcessor
 * @author xiemalin
 */
public class MessageListenerAnnotationResolver extends
		AbstractAnnotationParserCallback {

	private static final Logger LOGGER = LoggerFactory
			.getLogger(MessageListenerAnnotationResolver.class);

	/**
	 * cached ListenerSubscriber instances list
	 */
	private List<ListenerSubscriber> listenerSubscribers = new ArrayList<ListenerSubscriber>();

    /**
     * cached QueueListenerSubscriber instances list
     */
    private List<QueueListenerSubscriber> queueListenerSubscribers = new ArrayList<QueueListenerSubscriber>();

    /**
	 * cached bigpipe clients
	 */
	private List<SimpleNoneBlockingPubClient> clients = new ArrayList<SimpleNoneBlockingPubClient>();

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.baidu.bigpipe.spring.annotation.AnnotationParserCallback#
	 * annotationAtType(java.lang.annotation.Annotation, java.lang.Object,
	 * java.lang.String,
	 * org.springframework.beans.factory.config.ConfigurableListableBeanFactory)
	 */
	@Override
	public Object annotationAtType(Annotation t, Object bean, String beanName,
			ConfigurableListableBeanFactory beanFactory) throws BeansException {

		if (t instanceof MessageListener) {
			if (bean instanceof BigpipeMessageListener) {
				LOGGER.info("Annotation 'MessageListener' for target '"
						+ beanName
						+ "' detected and add to message listener callback.");

				ListenerSubscriber listenerSubscriber = doParseListenerSubscriber(
						(MessageListener) t, (BigpipeMessageListener) bean,
						beanName, beanFactory);
				listenerSubscribers.add(listenerSubscriber);

			} else {
				throw new BeanInitializationException(
						"Annotation 'MessageListener' for target '" + beanName
								+ "' is not implments "
								+ BigpipeMessageListener.class.getName());
			}

		}
        if (t instanceof QueueMessageListener) {
            if (bean instanceof PipeletIdAwareBigpipeMessageListener) {
                LOGGER.info("Annotation 'QueueMessageListener' for target '"
                        + beanName
                        + "' detected and add to queue message listener callback.");

                QueueListenerSubscriber queueListenerSubscriber = doParseQueueListenerSubscriber(
                        (QueueMessageListener) t, (PipeletIdAwareBigpipeMessageListener) bean,
                        beanName, beanFactory);
                queueListenerSubscribers.add(queueListenerSubscriber);

            } else {
                throw new BeanInitializationException(
                        "Annotation 'MessageListener' for target '" + beanName
                                + "' is not implments "
                                + BigpipeMessageListener.class.getName());
            }

        }

		return bean;
	}

	/**
	 * Parse annotation {@link MessageListener} and to build
	 * {@link DefaultListenerSubscriber} instance
	 * 
	 * @param t
	 *            {@link MessageListener} annotation
	 * @param bean
	 *            target {@link BigpipeMessageListener} bean
	 * @param beanName
	 *            the name of bean in spring context
	 * @param beanFactory
	 *            bean factory instance in spring context
	 * @return {@link ListenerSubscriber} instance.
	 */
	private ListenerSubscriber doParseListenerSubscriber(MessageListener t,
			BigpipeMessageListener bean, String beanName,
			ConfigurableListableBeanFactory beanFactory) {

		// try to get bigpipe conf
		String bigPipeConfBeanName = parsePlaceholder(t.bigPipeConf());
		BigPipeConf bigPipeConf = beanFactory.getBean(bigPipeConfBeanName,
				BigPipeConf.class);

		// do clone a new instance
		BigPipeConf eBigPipeConf = bigPipeConf.clone();

		if (!StringUtils.isEmpty(t.pipeletName())) {
			String pipeLetName = parsePlaceholder(t.pipeletName());
			eBigPipeConf.setPipeletName(pipeLetName);
		}

		if (!StringUtils.isEmpty(t.cluster())) {
			String cluster = parsePlaceholder(t.cluster());
			eBigPipeConf.setCluster(cluster);
		}

		if (!StringUtils.isEmpty(t.pipeletId())) {
			String pipeletId = parsePlaceholder(t.pipeletId());
			eBigPipeConf.setPipeletId(NumberUtils.parseNumber(pipeletId,
					int.class));
		}

		DefaultListenerSubscriber listenerSubscriber = new DefaultListenerSubscriber();
		listenerSubscriber.setConf(eBigPipeConf);
		if (!StringUtils.isEmpty(t.subcribePositionStore())) {
			String name = parsePlaceholder(t.subcribePositionStore());
			SubcribePositionStore subcribePositionStore = beanFactory.getBean(
					name, SubcribePositionStore.class);
			listenerSubscriber.setPositionStore(subcribePositionStore);
		}

		MessageBodyConverter messageBodyConverter;
		if (!StringUtils.isEmpty(t.bodyConverter())) {
			String bodyConverter = parsePlaceholder(t.bodyConverter());
			messageBodyConverter = beanFactory.getBean(bodyConverter,
					MessageBodyConverter.class);
		} else {
			messageBodyConverter = new StringMessageBodyConverter();
		}
		listenerSubscriber.setBodyConverter(messageBodyConverter);
		listenerSubscriber.setMessageListener(bean);

		// do initialize
		listenerSubscriber.init();
		return listenerSubscriber;

	}

    private QueueListenerSubscriber doParseQueueListenerSubscriber(QueueMessageListener t,
																   PipeletIdAwareBigpipeMessageListener bean, String beanName,
																   ConfigurableListableBeanFactory beanFactory) {

        // try to get bigpipe conf
        String bigPipeConfBeanName = parsePlaceholder(t.bigPipeConf());
        BigPipeConf bigPipeConf = beanFactory.getBean(bigPipeConfBeanName,
                BigPipeConf.class);

        // do clone a new instance
        BigPipeConf eBigPipeConf = bigPipeConf.clone();

        if (!StringUtils.isEmpty(t.pipeletName())) {
            String pipeLetName = parsePlaceholder(t.pipeletName());
            eBigPipeConf.setPipeletName(pipeLetName);
        }

        if (!StringUtils.isEmpty(t.cluster())) {
            String cluster = parsePlaceholder(t.cluster());
            eBigPipeConf.setCluster(cluster);
        }

//        if (!StringUtils.isEmpty(t.pipeletId())) {
//            String pipeletId = parsePlaceholder(t.pipeletId());
//            eBigPipeConf.setPipeletId(NumberUtils.parseNumber(pipeletId,
//                    int.class));
//        }

        DefaultQueueListenerSubscriber listenerSubscriber = new DefaultQueueListenerSubscriber();
        listenerSubscriber.setConf(eBigPipeConf);


        MessageBodyConverter messageBodyConverter;
        if (!StringUtils.isEmpty(t.bodyConverter())) {
            String bodyConverter = parsePlaceholder(t.bodyConverter());
            messageBodyConverter = beanFactory.getBean(bodyConverter,
                    MessageBodyConverter.class);
        } else {
            messageBodyConverter = new StringMessageBodyConverter();
        }
        listenerSubscriber.setBodyConverter(messageBodyConverter);
        listenerSubscriber.setMessageListener(bean);

        // do initialize
        listenerSubscriber.init();
        return listenerSubscriber;

    }

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.baidu.bigpipe.spring.annotation.AnnotationParserCallback#
	 * annotationAtTypeAfterStarted(java.lang.annotation.Annotation,
	 * java.lang.Object, java.lang.String,
	 * org.springframework.beans.factory.config.ConfigurableListableBeanFactory)
	 */
	@Override
	public void annotationAtTypeAfterStarted(Annotation t, Object bean,
			String beanName, ConfigurableListableBeanFactory beanFactory)
			throws BeansException {

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.baidu.bigpipe.spring.annotation.AnnotationParserCallback#
	 * annotationAtField(java.lang.annotation.Annotation, java.lang.Object,
	 * java.lang.String, org.springframework.beans.PropertyValues,
	 * org.springframework.beans.factory.config.ConfigurableListableBeanFactory,
	 * java.lang.reflect.Field)
	 */
	@Override
	public Object annotationAtField(Annotation t, Object value,
			String beanName, PropertyValues pvs,
			ConfigurableListableBeanFactory beanFactory, Field field)
			throws BeansException {
		if (t instanceof MessageSender) {

			SimpleNoneBlockingPubClient client = doParseMessageSenderAnnotation(
					(MessageSender) t, beanFactory);

			clients.add(client);
			return client;

		}

		return value;
	}

	/**
	 * Parse annotation {@link MessageListener} and to build
	 * {@link NoneBlockingBigpipePubClient} instance
	 * 
	 * @param t
	 *            {@link MessageSender} annotation
	 * @param beanFactory
	 *            bean factory instance in spring context
	 * @return new create {@link NoneBlockingBigpipePubClient} instance
	 */
	private SimpleNoneBlockingPubClient doParseMessageSenderAnnotation(MessageSender t,
			ConfigurableListableBeanFactory beanFactory) {

		// try to get bigpipe conf
		String bigPipeConfBeanName = parsePlaceholder(t.bigPipeConf());
		BigPipeConf bigPipeConf = beanFactory.getBean(bigPipeConfBeanName,
				BigPipeConf.class);

		// do clone a new instance
		BigPipeConf eBigPipeConf = bigPipeConf.clone();

		if (!StringUtils.isEmpty(t.pipeletName())) {
			String pipeLetName = parsePlaceholder(t.pipeletName());
			eBigPipeConf.setPipeletName(pipeLetName);
		}

		if (!StringUtils.isEmpty(t.cluster())) {
			String cluster = parsePlaceholder(t.cluster());
			eBigPipeConf.setCluster(cluster);
		}

		if (!StringUtils.isEmpty(t.pipeletId())) {
			String pipeletId = parsePlaceholder(t.pipeletId());
			eBigPipeConf.setPipeletId(NumberUtils.parseNumber(pipeletId,
					int.class));
		}

		SimpleNoneBlockingPubClient client = new SimpleNoneBlockingPubClient();
		client.setConf(eBigPipeConf);

		client.init();

		return client;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.baidu.bigpipe.spring.annotation.AnnotationParserCallback#
	 * annotationAtMethod(java.lang.annotation.Annotation, java.lang.Object,
	 * java.lang.String, org.springframework.beans.PropertyValues,
	 * org.springframework.beans.factory.config.ConfigurableListableBeanFactory,
	 * java.lang.reflect.Method)
	 */
	@Override
	public Object annotationAtMethod(Annotation t, Object value,
			String beanName, PropertyValues pvs,
			ConfigurableListableBeanFactory beanFactory, Method method)
			throws BeansException {

		return value;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.baidu.bigpipe.spring.annotation.AnnotationParserCallback#
	 * getTypeAnnotation()
	 */
	@Override
	public List<Class<? extends Annotation>> getTypeAnnotation() {
        List<Class<? extends Annotation>> annos = new ArrayList<Class<? extends Annotation>>();
        annos.add(MessageListener.class);
        annos.add(QueueMessageListener.class);
		return annos;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.baidu.bigpipe.spring.annotation.AnnotationParserCallback#
	 * getMethodFieldAnnotation()
	 */
	@Override
	public List<Class<? extends Annotation>> getMethodFieldAnnotation() {
		List<Class<? extends Annotation>> annos = new ArrayList<Class<? extends Annotation>>();
		annos.add(MessageSender.class);
		return annos;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.baidu.bigpipe.spring.annotation.AnnotationParserCallback#destroy()
	 */
	@Override
	public void destroy() throws Exception {
		for (ListenerSubscriber listenerSubscriber : listenerSubscribers) {
			try {
				listenerSubscriber.shutDown();
			} catch (Exception e) {
				LOGGER.error("error on shutdown listener subscriber due to :"
						+ e.getMessage(), e);
			}
		}
        for (QueueListenerSubscriber listenerSubscriber : queueListenerSubscribers) {
            try {
                listenerSubscriber.shutDown();
            } catch (Exception e) {
                LOGGER.error("error on shutdown listener subscriber due to :"
                        + e.getMessage(), e);
            }
        }

		for (SimpleNoneBlockingPubClient client : clients) {
			try {
				client.shutDown();
			} catch (Exception e) {
				LOGGER.error(
						"error on shutdown sender due to :" + e.getMessage(), e);
			}
		}
	}

}
