package com.baidu.bigpipe.transport;

import java.io.IOException;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import com.baidu.bigpipe.protocol.BigpipePacket;
import com.baidu.bigpipe.protocol.McpackCommand;
import com.baidu.bigpipe.protocol.QueueRequest;
import com.baidu.bigpipe.protocol.meta.exp.InvalidParameter;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.AclCommand;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.BigpipeCommand;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.BigpipeCommand.CommandType;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.ConnectCommand;
import com.baidu.mcpack.McpackException;
import com.google.protobuf.InvalidProtocolBufferException;

/**
 * 支持bigpipe session的网络流抽象
 *
 * @author hexiufeng
 */
public abstract class AbstractSessionSocketStream implements SessionSocketStream {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractSessionSocketStream.class);

    /**
     * 发布消息角色
     */
    public final static int BIGPIPE_ROLE_PUB = 4;
    /**
     * 订阅消息角色
     */
    public final static int BIGPIPE_ROLE_SUB = 5;

    @Override
    public void connectSession(PipeRuntime pipeRuntime, PipeletInfo pipeletInfo) throws IOException {
        connectSession(pipeRuntime, pipeletInfo, buildSocketIfNotExist(), getTransStrategy(), getReceiver(), getRole());
    }

    /**
     * 创建通信的socket对象。这里推荐缓存socket对象，如果没有现成的的socket对象 则创建
     *
     * @return 构建好的{@link Socket}对象
     */
    protected abstract Socket buildSocketIfNotExist();

    /**
     * 获取配置的传输控制策略
     *
     * @return {@link TransportStrategy}传输控制策略对象
     */
    protected abstract TransportStrategy getTransStrategy();

    /**
     * 返回配置的{@link Receiver}对象，该对象负责从{@link Socket}流中 读取数据
     *
     * @return {@link Receiver} 数据接收器对象
     */
    protected abstract Receiver getReceiver();

    /**
     * 获取构建bigpipe session的角色,返回以下取值:<br>
     * <ul>
     * <li>{@link AbstractSessionSocketStream#BIGPIPE_ROLE_PUB}=4 发布消息角色</li>
     * <li>{@link AbstractSessionSocketStream#BIGPIPE_ROLE_SUB}=5 订阅消息角色</li>
     * </ul>
     *
     * @return 构建session的角色
     */
    protected abstract int getRole();

    protected abstract void afterCreateSession() throws IOException;

    /**
     * 建立pipelet session,需要指明建立session的role:<br>
     * <ul>
     * <li>{@link AbstractSessionSocketStream#BIGPIPE_ROLE_PUB}=4 发布消息角色</li>
     * <li>{@link AbstractSessionSocketStream#BIGPIPE_ROLE_SUB}=5 订阅消息角色</li>
     * </ul>
     *
     * @param pipeRuntime   bigpipe运行时状态
     * @param pipeletInfo   pipelet的信息
     * @param socket        socket
     * @param transStrategy 传输策略
     * @param reciever      接收策略
     * @param role          bigpipe角色
     *
     * @throws IOException io异常
     */
    private void connectSession(PipeRuntime pipeRuntime, PipeletInfo pipeletInfo, Socket socket,
                                TransportStrategy transStrategy, Receiver reciever, int role) throws IOException {
        // 订阅queue的时候 啥都不干,在openStream之后 直接 afterCreateSession();即可 ,不需要发送订阅指令.
        if (!StringUtils.isEmpty(pipeRuntime.getQueueName())) {
            // 开始接受数据
            afterCreateSession();
            return;
        }
        // builder connect command
        BigpipeCommand.Builder cmd = BigpipeCommand.newBuilder();
        cmd.setType(BigpipeCommand.CommandType.BMQ_CONNECT);
        ConnectCommand.Builder connCmd = cmd.getConnectBuilder();
        connCmd.setRole(role);

        String oldSession = pipeRuntime.sessionIdProvider.getSessionId(false);
        connCmd.setSessionId(pipeRuntime.sessionIdProvider.getSessionId(pipeRuntime.refreshSessionId));
        if (pipeletInfo.getUserName() != null && !pipeletInfo.getUserName().equals("")) {
            connCmd.setTopicName(pipeRuntime.topicName);
            AclCommand.Builder aclCmd = connCmd.getAuthorityBuilder();
            aclCmd.setUsername(pipeletInfo.getUserName());
            aclCmd.setPassword(pipeletInfo.getPwd());
        } else {
            throw new InvalidParameter("username can not empty!");
        }
        BigpipeCommand command = cmd.build();
        ByteBuffer buf = transStrategy.buildSimpleCommand(command.toByteArray());
        socket.getOutputStream().write(buf.array(), 0, buf.limit());
        // read data using blocking mode
        ByteBuffer read = reciever.blockRecieve(socket);
        if (read == null) {
            throw new RuntimeException("connect session failed");
        }
        // 解析返回信息
        BigpipePacket connected = parseCommand(read);
        if (connected.getCommand().getType() != CommandType.BMQ_CONNECTED) {
            throw new RuntimeException(connected.getCommand().getError().toString() + ",[old:cur]--" + oldSession + ":"
                    + connCmd.getSessionId());
        }
        // 记录 next message id
        long serverMessageId = connected.getCommand().getConnected().getSessionMessageId();
        LOGGER.warn("server session messageid {},local messageid {}", serverMessageId, pipeRuntime.sessionMessageId);
        if (serverMessageId > pipeRuntime.sessionMessageId) {
            pipeRuntime.sessionMessageId = serverMessageId;
        }
        afterCreateSession();
    }

    /**
     * 解析读取到的返回信息
     *
     * @param read 读取到的数据
     *
     * @return bigpipe的数据包
     *
     * @throws InvalidProtocolBufferException 异常
     */
    public static BigpipePacket parseCommand(ByteBuffer read) throws InvalidProtocolBufferException {
        read.order(ByteOrder.LITTLE_ENDIAN);
        int length = read.getInt();
        byte[] ackBuf = new byte[length];
        read.get(ackBuf);
        BigpipeCommand ack = BigpipeCommand.parseFrom(ackBuf);

        BigpipePacket pack = new BigpipePacket();
        pack.setCommand(ack);
        if (read.remaining() > 0) {
            byte[] payload = new byte[read.remaining()];
            read.get(payload);
            pack.setPayload(payload);
        }
        return pack;
    }

}
