package com.baidu.bigpipe.transport;

import java.io.IOException;

import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.bigpipe.protocol.meta.NameService;
import com.baidu.bigpipe.protocol.meta.ZKMetaLoader;
import com.baidu.bigpipe.protocol.meta.concept.InetAddress;
import com.baidu.bigpipe.protocol.meta.concept.QueueAddress;
import com.baidu.bigpipe.protocol.meta.concept.TopicAddress;
import com.baidu.bigpipe.protocol.meta.exp.NameResolveException;
import com.baidu.bigpipe.protocol.meta.exp.QueueLocateException;
import com.baidu.bigpipe.transport.conf.BigPipeConf;
import com.baidu.bigpipe.transport.conf.SocketConf;

/**
 * bigpipe session相关抽象，包括配置信息维护、状态维护、session构建等。 该类被发布和订阅类继承
 *
 * @author hexiufeng
 */
public abstract class BigpipeSessionSupport {
    private static final Logger LOGGER = LoggerFactory.getLogger(BigpipeSessionSupport.class);
    private final PipeletInfo pipeletInfo = new PipeletInfo();
    /**
     * pipelet运行时信息
     */
    protected final PipeRuntime pipeRuntime = new PipeRuntime();
    /**
     * session的生命周期信息
     */
    protected final LifeController lifeController = new LifeController();
    /**
     * 从tcp channel接收数据的接收器
     */
    protected Receiver reciever = new NsHeadReciever();

    // 这个BigpipeSessionSupport可以再抽象一层,支持queue或者topic或者,等等
    public static enum PIPE_TYPE {
        TOPIC, QUEUE
    }

    // 0表示最原始的topic,1表示queue,由继承它的类来指定,默认是topic.
    // 本着尽量少改动代码的原则,
    // 1.所以如果type是topic,那么此类中的openStream的Address就是TopicAddress
    // 2.如果type是queue,那么此类中的openStream的Address其实就是QueueAddress
    private int type = PIPE_TYPE.TOPIC.ordinal();

    public int getType() {
        return type;
    }

    public void setType(int type) {
        this.type = type;
    }

    /**
     * 初始化bigpipe的配置信息以及开启发布或订阅线程
     *
     * @param conf {@link BigPipeConf} 对象，描述bigpipe的配置信息
     */
    public void init(BigPipeConf conf) {
        pipeletInfo.setPipe(conf.getPipe());
        pipeletInfo.setCluster(conf.getCluster());
        pipeletInfo.setPipeletId(conf.getPipeletId());
        pipeletInfo.setUserName(conf.getUserName());
        pipeletInfo.setPwd(conf.getPwd());
        pipeletInfo.setPipeletName(conf.getPipeletName());
        pipeletInfo.setQueue(conf.getQueue()); // set queue name

        ZKMetaLoader metaLoader = new ZKMetaLoader();

        metaLoader.setConnectString(conf.getMetaString());

        NameService ns = new NameService(conf.getCluster());
        ns.setMetaLoader(metaLoader);
        pipeRuntime.ns = ns;
        pipeRuntime.reConnectMaxTimes = conf.getReConnectMaxTimes();
        pipeRuntime.sessionIdProvider = conf.getSessionProvider();

        continueConfig(conf);

        start(conf);
    }

    /**
     * 开启发布或者订阅线程
     *
     * @param conf {@link BigPipeConf} 对象，配置信息
     */
    public abstract void start(final BigPipeConf conf);

    /**
     * 继续应用配置信息，允许子类从{@link BigPipeConf}获取私有配置信息并应用
     *
     * @param conf {@link BigPipeConf} 配置信息
     */
    protected abstract void continueConfig(BigPipeConf conf);

    /**
     * 从zk中根据pipeletName获取实例地址
     *
     * @param ns          {@link NameService}对象
     * @param pipeletOrQueueName pipelet Name or queue Name
     *
     * @return {@link InetAddress}
     *
     * @throws NameResolveException 根据名字解析{@link TopicAddress} {@link QueueAddress}异常
     * @throws KeeperException      zk异常
     */
    protected abstract InetAddress lookupAddr(NameService ns, String pipeletOrQueueName) throws NameResolveException,
            KeeperException, QueueLocateException;

    /**
     * 安全的关闭连接
     */
    protected abstract void safeCloseTcpConnect();

    /**
     * 连接bigpipe失败时线程sleep
     *
     * @param cnt sleep时间，ms
     */
    protected abstract void waitingForConnect(int cnt);

    /**
     * 打开网络流
     *
     * @param addr {@link InetAddress}地址
     * @param conf {@link SocketConf}网络相关配置
     *
     * @return {@link SessionSocketStream} 流
     *
     * @throws IOException 网络io异常
     */
    protected abstract SessionSocketStream openStream(final InetAddress addr, final SocketConf conf)
            throws IOException;

    /**
     * 发送或者订阅失败时，快速失败处理
     *
     * @param needBlocking 是否堵塞其他客户端操作，比如不能够再发布消息
     */
    protected abstract void handleFastFailed(boolean needBlocking);

    /**
     * 构建可用的tcp channel
     *
     * @param init 第一次构建connect，一般是spring的init方法中调用
     * @param conf socket配置信息
     */
    protected final void buildConnect(boolean init, SocketConf conf) {
        safeCloseTcpConnect();
        doConnect(init, conf);
    }

    /**
     * 与bigpipe建立session连接
     *
     * @param init 是否是初始化，如果是初始化，重连次数是5次，并且打出异常信息，线程中断
     * @param conf {@link SocketConf}网络配置
     */
    private void doConnect(boolean init, SocketConf conf) {
        NameService ns = pipeRuntime.ns;
        int cnt = 0;
        int connectSessionFailedCount = 0;

        InetAddress addr = null;
        String pipeletOrQueueName = getPipeletOrQueueName();
        int zkCount = 0;

        int maxReConnectTimes = pipeRuntime.reConnectMaxTimes;
        if (init) {
            maxReConnectTimes = pipeRuntime.reConnectMaxTimes > 5 ? 5 : pipeRuntime.reConnectMaxTimes;
            if (pipeRuntime.reConnectMaxTimes < 3600) {
                LOGGER.info("you should configure reConnectMaxTimes more than 3600");
            }
        }

        LOGGER.info("start to doConnect...");

        while (cnt < maxReConnectTimes) {
            if (lifeController.isShutDown()) {
                LOGGER.info("user set shut down when doConnect, bye.");
                return;
            }
            try {
                LOGGER.info("start to lookup for " + pipeletOrQueueName);
                addr = lookupAddr(ns, pipeletOrQueueName);
                LOGGER.info("after lookup for " + pipeletOrQueueName);
            } catch (NameResolveException e) {
                LOGGER.error("get address from zk error,may be invalid parameters.", e);
                waitingForConnect(cnt);
                LOGGER.info("get address from zk error, zkCount= {}", zkCount);
                zkCount++;
                continue;
            } catch (KeeperException e) {
                LOGGER.error("get address from zk error,may be lost zookeeper server.", e);
                waitingForConnect(cnt);
                LOGGER.info("may be lost zookeeper server,zkCount= {}", zkCount);
                zkCount++;
                continue;
            } catch (Exception e) {
                LOGGER.error("get address from zk error,unknown error.", e);
                waitingForConnect(cnt);
                LOGGER.info("unknown error,zkCount= {}", zkCount);
                zkCount++;
                continue;
            }
            SessionSocketStream sessionStream = null;
            if (addr != null) {
                try {
                    LOGGER.info("open socket " + pipeletOrQueueName);
                    sessionStream = openStream(addr, conf);
                    if (getType() == PIPE_TYPE.QUEUE.ordinal()) {
                        QueueAddress queueAddress = (QueueAddress) addr;
                        pipeRuntime.queueName = queueAddress.getQueueName();
                        pipeRuntime.topicName = null;

                    } else {

                        TopicAddress topicAddress = (TopicAddress) addr;
                        pipeRuntime.topicName = topicAddress.getStripe().getName();
                        pipeRuntime.queueName = null;
                    }
                    LOGGER.info("doconnect ok, opened socket,try times:" + cnt);
                } catch (IOException e) {
                    LOGGER.error("doConnect error. try times:" + cnt, e);
                    safeCloseTcpConnect();
                    LOGGER.info("socket error,sleep 200ms");
                    try {
                        Thread.sleep(200);
                    } catch (InterruptedException e1) {
                        // ignore
                    }
                }
                if (sessionStream != null) {
                    try {
                        LOGGER.info("hand shake with bigpipe. " + pipeletOrQueueName);
                        pipeRuntime.refreshSessionId = init ? true : connectSessionFailedCount > 0;
                        if (!init && pipeRuntime.refreshSessionId) {
                            LOGGER.info("It will refresh sesion id");
                        }
                        sessionStream.connectSession(pipeRuntime, pipeletInfo);
                        return;
                    } catch (Exception e) {
                        connectSessionFailedCount++;
                        LOGGER.error("create session error, cnt= " + cnt, e);
                        safeCloseTcpConnect();
                        LOGGER.info("create session,sleep 200ms");
                        try {
                            Thread.sleep(200);
                        } catch (InterruptedException e1) {
                            // ignore
                        }
                    }
                }
            }
            cnt++;
        }
        LOGGER.info("try to create bigpipe session failed for {} times,fast failed all request.", cnt);

        if (init) {
            throw new RuntimeException("connect failed .");
        } else {
            handleFastFailed(true);
        }
    }

    /**
     * 生成pipeliet的完全名称.
     * pipelet name + "_" + id.
     *
     * @return pipeliet的完全名称.
     */
    protected String getPipeletOrQueueName() {
        if (getType() == PIPE_TYPE.QUEUE.ordinal()) {
            return pipeletInfo.getQueue();
        }
        if (pipeletInfo.getPipeletName() != null && pipeletInfo.getPipeletName().length() > 0) {
            return pipeletInfo.getPipeletName();
        }
        return pipeletInfo.getPipe() + "_" + pipeletInfo.getPipeletId();
    }
}
