package com.baidu.bigpipe.transport.pub;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.bigpipe.exp.PushException;
import com.baidu.bigpipe.protocol.LogIdGen;
import com.baidu.bigpipe.protocol.SequenceLogIdGen;
import com.baidu.bigpipe.transport.conf.BigPipeConf;

/**
 * 异步消息发送的基类,它继承了AbstractNioSession,内部使用nio实现
 * 
 * @author hexiufeng
 * 
 */
public abstract class AbstractNioPublisher extends AbstractNioSession implements AsynchronousPublisher {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractNioPublisher.class);

    // ************消息发布相关*************************************
    /**
     * 消息发布的策略,用来控制如何生成消息发送任务,比如打包
     */
    protected PublishStrategy pubStrategy;
    /**
     * log id生成器s
     */
    protected LogIdGen idGen = new SequenceLogIdGen();
    // *********END消息发布相关************************************************


    /**
     * push消息到BlockingQueue的策略抽象,策略模式，被应用到 {@link com.baidu.bigpipe.transport.pub.AbstractNioPublisher#pushMessage
     * pushMessage}
     * 
     * @author hexiufeng
     * 
     */
    private static interface PushMessageHandler {
        /**
         * push消息到BlockingQueue，一次push的可能是一批消息
         */
        void push();

        /**
         * push一批消息时,这一批消息的总个数
         * 
         * @return
         */
        int getMessageListCount();
    }

    public PublishStrategy getPubStrategy() {
        return pubStrategy;
    }

    public void setPubStrategy(PublishStrategy pubStrategy) {
        this.pubStrategy = pubStrategy;
    }


    @Override
    public void applyMessageIdGen(LogIdGen idGen) {
        this.idGen = idGen;
    }

    @Override
    protected  void continueConfig(BigPipeConf conf){
        super.continueConfig(conf);
        pubStrategy.applySocketConf(conf);
    }

    @Override
    public SendFutrue publish(Message message) throws PushException {
        final Message m = message;
        final SendFutrueImpl futrue = new SendFutrueImpl();
        m.future = futrue;
        pushMessage(new PushMessageHandler() {
            @Override
            public void push() {
                pubStrategy.submitMessage(m);
            }

            @Override
            public int getMessageListCount() {
                return 1;
            }
        });
        return futrue;
    }

    @Override
    public SendFutrue publish(List<Message> message) throws PushException {
        final List<Message> m = message;
        final SendFutrueImpl futrue = new SendFutrueImpl();
        pushMessage(new PushMessageHandler() {
            @Override
            public void push() {
                pubStrategy.submitMessage(m, futrue);
            }

            @Override
            public int getMessageListCount() {
                return m.size();
            }
        });
        return futrue;
    }

    @Override
    protected void handleShutDown() {
        pubStrategy.handleShutDown(pipeRuntime.getSessionIdProvider());
        lifeController.getShutDownWait().countDown();
    }

    @Override
    public void shutDown() {
        super.shutDown();
    }

    @Override
    protected void waitingForConnect(int cnt) {
        LOGGER.info("get address from zk failed, sleep 500 ms.");
        try {
            Thread.sleep(500);
        } catch (InterruptedException e1) {
            // ignore
        }
    }



    /**
     * 检查当前是否有错误发生
     * 
     * @throws PushException PushException
     */
    private void checkErrOrShutdown() throws PushException {
        if (sessionRuntime.hasError || lifeController.isShutDown()) {
            throw new PushException(sessionRuntime.hasError ? "error." : "shutdown.");
        }
    }

    /**
     * 把消息push到queue中，策略模式, {@link com.baidu.bigpipe.transport.pub.AbstractNioPublisher.PushMessageHandler
     * PushMessageHandler} 提供具体的push实现
     * 
     * @param handler 控制器
     * @throws PushException PushException
     */
    private void pushMessage(PushMessageHandler handler) throws PushException {
        checkErrOrShutdown();
        try {
            pubStrategy.accqireToken(handler.getMessageListCount());
            checkErrOrShutdown();
            handler.push();
        } catch (InterruptedException e) {
            throw new PushException();
        } catch (PushException e) {
            pubStrategy.releaseToken(handler.getMessageListCount());
            throw e;
        }

        if (wakenUp.compareAndSet(false, true)) {
            selector.wakeup();
        }
    }
}
