package com.baidu.bigpipe.transport.pub;

import java.io.IOException;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.bigpipe.protocol.meta.NameService;
import com.baidu.bigpipe.protocol.meta.concept.TopicAddress;
import com.baidu.bigpipe.protocol.meta.exp.NameResolveException;
import com.baidu.bigpipe.transport.BigpipeSessionSupport;
import com.baidu.bigpipe.transport.Receiver;
import com.baidu.bigpipe.transport.conf.BigPipeConf;
import com.baidu.bigpipe.transport.conf.SocketConf;
import com.baidu.bigpipe.transport.pub.context.ReadContext;
import com.baidu.bigpipe.transport.pub.context.ReadState;
import com.baidu.bigpipe.transport.pub.context.WriteState;
import com.baidu.bigpipe.transport.pub.context.WriteTask;

/**
 * 基于nio单线程单tcp通信的基类，它实现了大部分与通信相关的逻辑，比如构建tcpchannel，selector，开启线程， 发送接收数据等。采用模板方法设计模式，与业务相关的逻辑被抽象成各种抽象方法，延迟到具体子类实现。
 * 
 * @author hexiufeng
 * 
 */
public abstract class AbstractNioSession extends BigpipeSessionSupport {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractNioSession.class);
    /**
     * 是否采用tcp的双工模式
     */
    protected boolean duplexMode = true;

    // *********END消息发布相关************************************************

    // ************nio selector相关********************************
    protected volatile Selector selector;
    // 初始化selector
    {
        try {
            this.selector = Selector.open();
        } catch (IOException e) {
            LOGGER.error("init failed.", e);
            throw new RuntimeException(e);
        }
    }
    /**
     * 重建selector标记，防止重建期间monitor线程调用selector.wakeup
     */
    private AtomicBoolean reBuilding = new AtomicBoolean();
    /**
     * copy from netty, 用于防止频繁的selector.wakeup
     */
    protected AtomicBoolean wakenUp = new AtomicBoolean();
    /**
     * selector在linux高并发下有bug，是否要避免这个bug
     */
    private boolean avoidEpollBug = true;
    protected volatile int selecttimeout = 500;
    protected int publisherTryMaxCount = 3;
    // ************END nio selector相关********************************

    /**
     * 当前session的运行时状态
     */
    protected final SessionRuntime sessionRuntime = new SessionRuntime();

    // 当前物理tcp连接
    protected volatile SocketChannel tcpConnect;
    /**
     * socket连接的配置信息
     */
    protected volatile SocketConf socketConf;

    /**
     * 记录当前session运行时的状态
     * 
     * @author hexiufeng
     * 
     */
    static class SessionRuntime {
        /**
         * 是否需要等待新任务，如果需要新任务，则需要从消息队列提取消息并转换为新任务
         */
        boolean waitTask = true;
        /**
         * 是否出现错误
         */
        volatile boolean hasError = false;
        /**
         * 是否tcp已经无效，需要重新打开一个tcp
         */
        boolean needOpenTcp = false;
    }

    /**
     * 绑定在SocketKey上的任务占位符,它作为SocketKey的attachment而存在，是可以被重用的， 它包含需要发送的任务和需要被接收数据的上下文，在双工模式下二者是没有任何关联关系的
     * 
     * @author hexiufeng
     * 
     */
    protected static class AttachHolder {
        /**
         * 当前需要被处理的任务
         */
        WriteTask writeTask;
        /**
         * 任务被发送后，接收消息的上下文
         */
        ReadContext rc;
    }

    public SocketConf getSocketConf() {
        return socketConf;
    }

    public void setSocketConf(SocketConf socketConf) {
        this.socketConf = socketConf;
    }

    public Receiver getReciever() {
        return reciever;
    }

    public void setReciever(Receiver reciever) {
        this.reciever = reciever;
    }

    public boolean isDuplexMode() {
        return duplexMode;
    }

    public void setDuplexMode(boolean duplexMode) {
        this.duplexMode = duplexMode;
    }

    public int getPublisherTryMaxCount() {
        return publisherTryMaxCount;
    }

    public void setPublisherTryMaxCount(int publisherTryMaxCount) {
        this.publisherTryMaxCount = publisherTryMaxCount;
    }

    /**
     * 关闭当前session，如果有未执行完成的任务，会放弃执行并返回给调用方执行失败信息
     */
    public void shutDown() {
        lifeController.setShutDown(true);
        try {
            lifeController.getShutDownWait().await(socketConf.getShutDownTimeout(), TimeUnit.MINUTES);
            try {
                this.selector.close();
            } catch (IOException e) {
                LOGGER.error("shutdown error, ignore.", e);
            }
            this.safeCloseTcpConnect();
        } catch (InterruptedException e) {
            // ignore
        }
    }

    @Override
    protected void continueConfig(BigPipeConf conf) {
        this.selecttimeout = conf.getSelecttimeout();
    }

    @Override
    protected TopicAddress lookupAddr(NameService ns, String pipeletName) throws NameResolveException, KeeperException {
        return ns.lookupForPub(pipeletName);
    }

    /**
     * 当读写tcp出现异常时调用该方法来处理异常，一般来说如果有的任务处理出现异常，需要把任务恢复初始状态 然后关闭对应的channel
     * 
     * @param key
     */
    protected abstract void handleSelectorException(final SelectionKey key);

    /**
     * 探测当前是否有任务已经timeout，如果有需要处理这些超时任务
     */
    protected abstract void handleTimeout();

    /**
     * 向tcp channel中写入数据
     * 
     * @param k {@link SelectionKey}对象
     * @return {@link WriteState}
     */
    protected abstract WriteState write(SelectionKey k);

    /**
     * 从tcp channel中读取数据
     * 
     * @param k {@link SelectionKey}对象
     * @return {@link ReadState}
     */
    protected abstract ReadState read(SelectionKey k);

    /**
     * 获取下一个需要处理的新任务
     * 
     * @return {@link WriteTask}
     */
    protected abstract WriteTask startNewTask();

    /**
     * config Task
     * 
     * @param task task
     * @param isNeedTcp isNeedTcp
     */
    protected abstract void configTask(WriteTask task, boolean isNeedTcp);

    /**
     * 保证构建tcp连接
     */
    protected abstract void ensureTcp();

    /**
     * 当任何调用方调用 shutdown 方法后，需要做的善后工作
     */
    protected abstract void handleShutDown();

    
   /**
    * 开启selector.select线程
    * 
    * @param conf 配置
    */
    public final void start(BigPipeConf conf) {
        socketConf = conf;
        // 在监控线程启动前首先构建tcp channel,监控线程启动后会检测到该channel,后续由监控线程来
        // 重新构建channel，此过程即使tcp channel不是volatile修饰的也能保证线程安全
        buildConnect(true, conf);

        Thread t = new Thread(new Runnable() {

            @Override
            public void run() {
                runningFunction();
            }

        });

        String threadName = "bigpipe-client-thread-";
        if (conf.getThreadName() != null) {
            threadName = threadName + conf.getThreadName() + "-";
        }
        t.setName(threadName + t.getId());
        t.start();
        try {
            lifeController.getThreadRunning().await();
        } catch (InterruptedException e) {
            LOGGER.error("start thread error.", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 封装关闭tcp channel，隐藏异常处理细节
     */
    @Override
    protected final void safeCloseTcpConnect() {
        if (tcpConnect == null) {
            return;
        }
        try {
            tcpConnect.configureBlocking(false);
            tcpConnect.close();
        } catch (IOException e) {
            LOGGER.info("safeCloseTcpConnect", e);
        } finally {
            tcpConnect = null;
        }
    }

    /**
     * 线程执行核心方法，该方法会监控selector，重新构建selector，监控tcp状态,处理写入、读取数据和其他业务处理
     */
    private void runningFunction() {
        // ****copy from netty************************************
        int selectReturnsImmediately = 0;
        // use 80% of the timeout for measure
        final long minSelectTimeout = TimeUnit.MILLISECONDS.toNanos(selecttimeout) * 80 / 100;
        boolean wakenupFromLoop = false;
        // *****copy from netty********************************************
        lifeController.getThreadRunning().countDown();
        while (true) {
            wakenUp.set(false);

            int selected = 0;
            long beforeSelect = System.nanoTime();
            try {
                selected = this.selector.select(selecttimeout);
            } catch (ClosedSelectorException ce) {
                throw new RuntimeException(ce);
            } catch (CancelledKeyException e) {
                // maybe jdk epoll bug
                LOGGER.info("maybe jdk epoll bug", e);
                selected = -1;
            } catch (IOException e) {
                LOGGER.info("selecor.select", e);
            }
            // copy from netty to prevent jdk1.6 epoll bug on os linux
            if (this.avoidEpollBug && selected == 0 && !wakenupFromLoop && !wakenUp.get()) {
                long timeBlocked = System.nanoTime() - beforeSelect;

                if (timeBlocked < minSelectTimeout) {
                    boolean notConnected = false;
                    // loop over all keys as the selector may was unblocked
                    // because of a closed channel
                    for (SelectionKey key : selector.keys()) {
                        SelectableChannel ch = key.channel();
                        try {
                            if (ch instanceof DatagramChannel && !((DatagramChannel) ch).isConnected()
                                    || ch instanceof SocketChannel && !((SocketChannel) ch).isConnected()) {
                                notConnected = true;
                                // cancel the key just to be on the safe side
                                key.cancel();
                            }
                        } catch (CancelledKeyException e) {
                            // ignore
                        }
                    }
                    if (notConnected) {
                        selectReturnsImmediately = 0;
                    } else {
                        // returned before the minSelectTimeout elapsed with
                        // nothing select.
                        // this may be the cause of the jdk epoll(..) bug, so
                        // increment the counter
                        // which we use later to see if its really the jdk bug.
                        selectReturnsImmediately++;
                    }
                } else {
                    selectReturnsImmediately = 0;
                }
                // copy from netty
                if (selectReturnsImmediately == 1024) {
                    // The selector returned immediately for 10 times in a row,
                    // so recreate one selector as it seems like we hit the
                    // famous epoll(..) jdk bug.
                    this.rebuildSeletor();
                    selectReturnsImmediately = 0;
                    wakenupFromLoop = false;
                    // try to select again
                    continue;
                }
            } else {
                // reset counter
                selectReturnsImmediately = 0;
            }

            // copy from netty,由于selector.wakeu消耗比较大,需要降低wakeup的频率
            if (wakenUp.get()) {
                wakenupFromLoop = true;
                selector.wakeup();
            } else {
                wakenupFromLoop = false;
            }
            // if (selected == 0 && !wakenUp.get()) {
            // LOGGER.info("selector wait timeout.");
            // };
            // process normal task
            if (selected > 0) {
                Set<SelectionKey> keys = this.selector.selectedKeys();
                for (Iterator<SelectionKey> i = keys.iterator(); i.hasNext();) {
                    SelectionKey k = i.next();
                    i.remove();
                    if (k.isReadable()) {
                        ReadState state = readCore(k);
                        if (state == ReadState.ReadEOF || state == ReadState.Error) {
                            break;
                        }
                    }
                    if (k.isWritable()) {
                        WriteState state = writeCore(k);
                        if (state == WriteState.Error) {
                            break;
                        }
                    }
                }
                if (keys.size() > 0) {
                    keys.clear();
                }
                if (lifeController.isShutDown()) {
                    break;
                }
            }
            if (lifeController.isShutDown()) {
                handleShutDown();
                break;
            }
            this.handleTimeout();
            if (this.tcpConnect == null) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("tcp is null, there is not task, need to wait new task.");
                }
                // buildConnect(false, socketConf);
                sessionRuntime.waitTask = true;
            }
            if (lifeController.isShutDown()) {
                handleShutDown();
                break;
            }
            if (sessionRuntime.waitTask) {
                tryToStartNewTask();
            }
        }
    }

    /**
     * 尝试获取下一个需要处理的任务，并且开时处理这个任务
     */
    protected void tryToStartNewTask() {
        // bigpipe在60秒内没有收到数据会自动断开连接，在监控到断开后没有必要马上连接,
        // 因为如果没有新的任务,60秒后它又被自动断开，因此把重新连接推迟到有新任务时
        // 资源消耗会更少,尤其是对bigpipe服务端资源的消耗，当然重连会降低首个任务处理的性能，
        // 但当任务量不大时这种性能损耗是可容忍的，如果任务量很大，后续任务会及时被处理，这时
        // 连接是不会被server side断开的，所以性能依旧不是问题，综合权衡, 我们选择降低bigpipe的损耗。
        boolean reConnect = sessionRuntime.needOpenTcp;
        WriteTask newTask = startNewTask();
        if (newTask != null) {
            if (reConnect) {
                ensureTcp();
                sessionRuntime.needOpenTcp = false;
            }
            configTask(newTask, reConnect);
            SelectionKey k = tcpConnect.keyFor(selector);
            AttachHolder holder = (AttachHolder) k.attachment();
            if (holder == null) {
                holder = new AttachHolder();
                k.attach(holder);
            }
            holder.writeTask = newTask;
            if (reConnect) {
                k.interestOps((k.interestOps() & (~SelectionKey.OP_READ)));
            }
            setupSelectionKeyInterest(k);
            sessionRuntime.waitTask = false;
        }
    }

    /**
     * 设置selectionkey的interest
     * 
     * @param k {@link SelectionKey}
     */
    private void setupSelectionKeyInterest(SelectionKey k) {
        if (this.duplexMode) {
            k.interestOps(k.interestOps() | SelectionKey.OP_WRITE);
        } else {
            k.interestOps(SelectionKey.OP_WRITE);
        }
    }

    /**
     * 从SocketChannel中读取数据,可能只读取一部分，然后根据状态进行下一步的操作
     * 
     * @param k {@link SelectionKey}
     * @return {@link ReadState}
     */
    private ReadState readCore(SelectionKey k) {
        ReadState state = read(k);
        if (state == ReadState.ReadEOF) {
            LOGGER.info("read eof,close tcp, waiting next connect...");
            safeCloseTcpConnect();
            sessionRuntime.needOpenTcp = true;
        }
        if (state == ReadState.Finish) {
            if (!duplexMode) {
                sessionRuntime.waitTask = true;
            }
        }
        if (state == ReadState.Error) {
            LOGGER.info("read error, and close tcp, waiting next connect...");
            // buildConnect(false, socketConf);
            safeCloseTcpConnect();
            sessionRuntime.needOpenTcp = true;
        }
        return state;
    }

    /**
     * 向SocketChannel中写入数据,可能只写入一部分，然后根据状态进行下一步的操作
     * 
     * @param k {@link SelectionKey}
     * @return {@link WriteState}
     */
    private WriteState writeCore(SelectionKey k) {
        WriteState state = write(k);
        if (state == WriteState.NoTask) {
            sessionRuntime.waitTask = true;
            k.interestOps(k.interestOps() & (~SelectionKey.OP_WRITE));
        }
        if (state == WriteState.Finish) {
            if (!duplexMode) {
                k.interestOps(SelectionKey.OP_READ);
            } else {
                sessionRuntime.waitTask = true;
                k.interestOps(k.interestOps() | SelectionKey.OP_READ);
            }
        }
        return state;
    }

    /**
     * copy from netty to prevent jdk epoll bug
     */
    private void rebuildSeletor() {
        final Selector oldSelector = selector;
        final Selector newSelector;

        try {
            newSelector = Selector.open();
        } catch (Exception e) {
            LOGGER.info("rebuildSeletor", e);
            return;
        }
        // 调度monitor线程，防止重建selector期间monitor线程调用wake up
        reBuilding.set(true);
        // Register all channels to the new Selector.
        // int nChannels = 0;
        for (;;) {
            try {
                for (SelectionKey key : oldSelector.keys()) {
                    try {
                        if (key.channel().keyFor(newSelector) != null) {
                            continue;
                        }

                        int interestOps = key.interestOps();
                        key.cancel();
                        key.channel().register(newSelector, interestOps, key.attachment());
                        // nChannels ++;
                    } catch (Exception e) {
                        LOGGER.error("rebuildSeletor", e);
                        handleSelectorException(key);
                    }
                }
            } catch (ConcurrentModificationException e) {
                // Probably due to concurrent modification of the key set.
                continue;
            }

            break;
        }
        // 调度monitor线程
        synchronized (this) {
            selector = newSelector;
        }
        reBuilding.set(false);

        try {
            // time to close the old selector as everything else is registered
            // to the new one
            oldSelector.close();
        } catch (Throwable t) {
            LOGGER.error("oldSelector.close", t);
        }
    }
}
