package com.baidu.bigpipe.transport.pub;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.bigpipe.protocol.SessionIdProvider;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.BigpipeCommand;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.BigpipeCommand.CommandType;
import com.baidu.bigpipe.transport.NHeadTransportStrategy;
import com.google.protobuf.InvalidProtocolBufferException;

/**
 * 基于NsHead协议通讯的{@link com.baidu.bigpipe.transport.pub.PublishStrategy PublishStrategy}实现
 * 
 * @author hexiufeng
 * 
 */
public abstract class AbstractPublishStrategy extends NHeadTransportStrategy implements PublishStrategy {
    private static final Logger MESSAGEID_LOGGER = LoggerFactory.getLogger("bigpipe.topic.messageid.log");
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractPublishStrategy.class);

    /**
     * 内存中缓存的需要处理任务的上限
     */
    private int taskLimitCount = 20000;
    /**
     * 使用信号量来控制边界
     */
    private Semaphore taskLimit = new Semaphore(20000);
    /**
     * 用于缓存需要重新发送任务的queue，单线程内使用
     */
    // protected final Queue<WriteTask> repeatTaskList = new
    // LinkedList<WriteTask>();
    /**
     * 双工模式下允许的最大并发任务数
     */
    private int maxConcurrent = 1;
    /**
     * 当前的并任务数
     */
    private int conCurrent = 0;
    /**
     * 获取push任务许可的超时配置,单位ms
     */
    private int acquireTimeout = 5000;

    public int getTaskLimitCount() {
        return taskLimitCount;
    }

    public void setTaskLimitCount(int taskLimitCount) {
        this.taskLimitCount = taskLimitCount;
        taskLimit = new Semaphore(taskLimitCount);
    }

    public int getMaxConcurrent() {
        return maxConcurrent;
    }

    public void setMaxConcurrent(int maxConcurrent) {
        this.maxConcurrent = maxConcurrent;
    }

    @Override
    public void accqireToken() throws InterruptedException {
        if (!taskLimit.tryAcquire(acquireTimeout, TimeUnit.MILLISECONDS)) {
            throw new RuntimeException("wait to pub timeout");
        }
    }

    @Override
    public void accqireToken(int n) throws InterruptedException {
        if (!taskLimit.tryAcquire(n, acquireTimeout, TimeUnit.MILLISECONDS)) {
            throw new RuntimeException("wait to pub timeout");
        }
    }

    @Override
    public void releaseToken(int n) {
        taskLimit.release(n);
    }

    @Override
    public void releaseToken() {
        taskLimit.release();
    }

    @Override
    public void finishPub(ByteBuffer buf, SessionIdProvider sessionIdProvider) {
        buf.order(ByteOrder.LITTLE_ENDIAN);
        int length = buf.getInt();
        byte[] ackBuf = new byte[length];
        buf.get(ackBuf);
        String logId = null;
        long status = 0;
        try {
            BigpipeCommand ack = BigpipeCommand.parseFrom(ackBuf);
            if (ack.getType() != CommandType.BMQ_ACK) {
                LOGGER.error(ack.getError().toString());
                fastFailedRunning(sessionIdProvider);
            }
            status = ack.getAck().getStatus();
            logId = ack.getAck().getReceiptId();
            MESSAGEID_LOGGER.debug("published message position: {}", ack.getAck().getTopicMessageId());
        } catch (InvalidProtocolBufferException e) {
            // ignore
            fastFailedRunning(sessionIdProvider);
            return;
        }
        handleFinish(logId, status, sessionIdProvider);
    }

    /**
     * 出现异常时，快速失败处理
     * 
     * @param sessionIdProvider SessionIdProvider
     */
    protected abstract void fastFailedRunning(SessionIdProvider sessionIdProvider);

    /**
     * 消息发送完成或失败时控制后续处理，一般需要通知调用方
     * 
     * @param logId logId
     * @param status 状态
     * @param sessionIdProvider {@link SessionIdProvider}
     */
    protected abstract void handleFinish(String logId, long status, SessionIdProvider sessionIdProvider);

    /**
     * 当前并发任务是否达到最大值，是否可以继续处理新任务
     * 
     * @return 是否可以继续运行任务
     */
    protected boolean canRunTask() {
        return (conCurrent < maxConcurrent);
    }

    /**
     * 增加并发数，单线程
     */
    protected void registerRunTask() {
        conCurrent++;
    }

    /**
     * 减少并发数，单线程
     */
    protected void unRegisterRunTask() {
        conCurrent--;
    }
}
