package com.baidu.bigpipe.transport.pub;

import java.io.IOException;
import java.net.Socket;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.bigpipe.protocol.meta.concept.InetAddress;
import com.baidu.bigpipe.transport.AbstractSessionSocketStream;
import com.baidu.bigpipe.transport.PipeRuntime;
import com.baidu.bigpipe.transport.PipeletInfo;
import com.baidu.bigpipe.transport.Receiver;
import com.baidu.bigpipe.transport.SessionSocketStream;
import com.baidu.bigpipe.transport.TransportStrategy;
import com.baidu.bigpipe.transport.conf.SocketConf;
import com.baidu.bigpipe.transport.pub.context.ReadContext;
import com.baidu.bigpipe.transport.pub.context.ReadState;
import com.baidu.bigpipe.transport.pub.context.WriteState;
import com.baidu.bigpipe.transport.pub.context.WriteTask;

/**
 * 异步发送消息{@link com.baidu.bigpipe.transport.pub.AsynchronousPublisher AsynchronousPublisher}接口的实现类
 * 
 * @author hexiufeng
 * 
 */
public class AsynchronousPublisherImpl extends AbstractNioPublisher implements AsynchronousPublisher {
    private static final Logger LOGGER = LoggerFactory.getLogger(AsynchronousPublisherImpl.class);

    @Override
    protected void handleSelectorException(SelectionKey key) {
        super.sessionRuntime.hasError = true;
        super.safeCloseTcpConnect();
        pubStrategy.fastFailed(pipeRuntime.getSessionIdProvider());
        sessionRuntime.needOpenTcp = true;
        super.sessionRuntime.hasError = false;
    }

    @Override
    protected WriteState write(SelectionKey k) {
        AttachHolder holder = (AttachHolder) k.attachment();
        if (holder == null || holder.writeTask == null) {
            return WriteState.NoTask;
        }
        SocketChannel chnl = (SocketChannel) k.channel();
        try {
            chnl.write(holder.writeTask.getBuf());
            // 修改状态，不再算作重发状态
            if (holder.writeTask.isReConnectAndNoSend()) {
                holder.writeTask.setReConnectAndNoSend(false);
            }
            if (holder.writeTask.getBuf().remaining() == 0) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("write finish for logid {},session id is {} ", holder.writeTask.getLogId(),
                            holder.writeTask.getSessionMessageId());
                }
                holder.writeTask = null;
                return WriteState.Finish;
            }
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("writing for logid {},session id is {} ", holder.writeTask.getLogId(),
                        holder.writeTask.getSessionMessageId());
            }
            return WriteState.Writing;
        } catch (IOException e) {
            LOGGER.info("write", e);
            handleSelectorException(k);
            return WriteState.Error;
        }
    }

    @Override
    protected ReadState read(SelectionKey k) {
        AttachHolder holder = (AttachHolder) k.attachment();
        SocketChannel chnl = (SocketChannel) k.channel();
        if (holder.rc == null) {
            holder.rc = new ReadContext();
        }
        try {
            reciever.recieve(chnl, holder.rc);
            if (holder.rc.isEOF()) {
                LOGGER.info("recieve data with eof!!!");
                handleFastFailed(true);
                holder.rc = null;
                holder.writeTask = null;
                return ReadState.ReadEOF;
            }
            if (holder.rc.isComplete()) {
                holder.rc.getBuf().flip();
                pubStrategy.finishPub(holder.rc.getBuf(), pipeRuntime.getSessionIdProvider());
                holder.rc = null;
                return ReadState.Finish;
            }
        } catch (IOException e) {
            LOGGER.info("read", e);
            handleFastFailed(true);
            holder.rc = null;
            holder.writeTask = null;
            return ReadState.Error;
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("reading for logid {},session id is {} ", holder.writeTask.getLogId(),
                    holder.writeTask.getSessionMessageId());
        }
        return ReadState.Reading;
    }

    @Override
    protected WriteTask startNewTask() {
        long msgId = super.pipeRuntime.getSessionMessageId();
        msgId++;
        WriteTask task =
                pubStrategy.getNextTask(idGen, msgId, pipeRuntime.getSessionIdProvider().getSessionId(false),
                        pipeRuntime.getTopicName());
        if (task != null) {
            if (task.getCount() == 0) {
                super.pipeRuntime.setSessionMessageId(msgId);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("new task messageid {}", task.getSessionMessageId());
                }
            } else {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("repeat task messageid {}", task.getSessionMessageId());
                }
            }
            return task;
        }
        return null;
    }

    @Override
    protected void ensureTcp() {
        buildConnect(false, super.socketConf);
    }

    @Override
    protected void configTask(WriteTask task, boolean isNeedTcp) {
        if (!isNeedTcp) {
            if (pubStrategy.getCurrentTaskCount() == 1) {
                task.setReConnectAndNoSend(true);
            }
        }
    }

    @Override
    protected void handleFastFailed(boolean needBlocking) {
        if (needBlocking) {
            super.sessionRuntime.hasError = true;
        }

        super.safeCloseTcpConnect();
        pubStrategy.fastFailed(pipeRuntime.getSessionIdProvider());

        if (needBlocking) {
            super.sessionRuntime.hasError = false;
        }
    }

    @Override
    protected void handleTimeout() {
        if (pubStrategy.handlePubTimeout(pipeRuntime.getSessionIdProvider())) {
            LOGGER.info("send time out, and close socket....");
            safeCloseTcpConnect();
            sessionRuntime.needOpenTcp = true;
        }
    }

    @Override
    protected SessionSocketStream openStream(final InetAddress addr, final SocketConf conf) throws IOException {
        
        this.tcpConnect = SocketChannel.open();
        // 使用SocketChannel.socket()方法获取socket对象后去连接，这是因为连接时需要设置连接超时，
        // 如果没有连接超时当网络环境不好的情况下线程有可能会在connect时被堵塞，导致不能及时切换到
        // pipelet的其他实例,另外，SocketChannel即使在堵塞模式下并且设置SoTimeout在读写时也不会
        // 有超时功能，这是因为SocketChannel是专为异步设计的，SoTimeout是为同步设计，SocketChannel
        // 和
        // SocketChannel.socket()在底层虽然使用相同的操作系统socket句柄，但在上层的封装是完全不同的，
        // 同时SocketChannel.socket()虽然也是socket类型对象，和传统Socket也具有相同的api，
        // 但它和传统的Socket在底层是完全不同的
        LOGGER.warn("stripe address--[{}:{}]", addr.getAddress().getHostName(), addr.getAddress().getPort());
        tcpConnect.configureBlocking(true);
        tcpConnect.socket().connect(addr.getAddress(), conf.getConectTimeout());
        tcpConnect.socket().setSoTimeout(conf.getIoTimeout());
        return new AbstractSessionSocketStream(){

            @Override
            public void connectSession(PipeRuntime pipeRuntime, PipeletInfo pipeletInfo) throws IOException {
                super.connectSession(pipeRuntime, pipeletInfo);
                tcpConnect.configureBlocking(false);
                LOGGER.info("register socket to nio. " + pipeletInfo.getPipeletName());
                SelectionKey key = tcpConnect.register(selector, SelectionKey.OP_READ);
                key.attach(new AttachHolder());
                sessionRuntime.waitTask = true;
            }

            @Override
            protected Socket buildSocketIfNotExist() {
                return tcpConnect.socket();
            }

            @Override
            protected TransportStrategy getTransStrategy() {
                return pubStrategy;
            }

            @Override
            protected Receiver getReceiver() {
                return reciever;
            }

            @Override
            protected int getRole() {
                return BIGPIPE_ROLE_PUB;
            }

            @Override
            protected void afterCreateSession() throws IOException {
                // nop
            }
            
        };
    }
}
