package com.baidu.bigpipe.transport.pub;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.zip.Adler32;

import com.baidu.bigpipe.protocol.LogIdGen;
import com.baidu.bigpipe.protocol.SessionIdProvider;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.BigpipeCommand;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.MessageCommand;
import com.baidu.bigpipe.transport.NSHead;
import com.baidu.bigpipe.transport.pub.context.WriteTask;

/**
 * 分批发送消息策略实现，组内消息可以使用双工模式，组与组之间顺序处理， 这要求组内消息的发送没有顺序要求
 * 
 * @author hexiufeng
 * 
 */
public class GroupPublishStrategy extends AbstractPublishStrategy implements PublishStrategy {
    /**
     * 保存需要发送的任务
     */
    private BlockingQueue<PubTask> taskQueue = new LinkedBlockingQueue<PubTask>();
    /**
     * 真正发送中的任务
     */
    private Map<String, PubTask> runingMap = new LinkedHashMap<String, PubTask>();
    
    // 初始化任务queue的容量
    {
        super.setMaxConcurrent(2000);
    }

    /**
     * 需要处理的任务的描述,指正在发送的任务
     * 
     * @author hexiufeng
     * 
     */
    private static class PubTask {
        /**
         * 发送一批消息时，一个发送任务包含多个消息
         */
        List<Message> list = new LinkedList<Message>();
        /**
         * 触发调用方不再堵塞
         */
        InternalFutrue futrue;
        /**
         * 整个任务的log Id
         */
        String logId;
        int index = 0;
        /**
         * 已经完成的消息logId
         */
        List<String> finList = new LinkedList<String>();
        /**
         * 监控信息
         */
        long pubTime;
        long writeTime;
        // long finshTime;
    }

    @Override
    public void submitMessage(Message msg) {
        PubTask pt = new PubTask();
        pt.pubTime = System.currentTimeMillis();
        pt.list.add(msg);
        pt.futrue = msg.future;
        pt.logId = msg.getLogId();
        taskQueue.offer(pt);
    }

    @Override
    public void submitMessage(List<Message> msgList, InternalFutrue futrue) {
        PubTask pt = new PubTask();
        pt.pubTime = System.currentTimeMillis();
        pt.list = msgList;
        pt.futrue = futrue;
        taskQueue.offer(pt);
    }

    @Override
    public WriteTask getNextTask(LogIdGen idGen, long messageId, String sessionId, String topicName) {
        // if(super.repeatTaskList.size() > 0){
        // WriteTask wt = super.repeatTaskList.poll();
        // return wt;
        // }
        Message msg = null;
        String groupLogId = null;
        for (String logId : runingMap.keySet()) {
            PubTask pt = runingMap.get(logId);
            if (pt.index < pt.list.size()) {
                groupLogId = pt.logId;
                msg = pt.list.get(pt.index);
                pt.index++;
                pt.writeTime = System.currentTimeMillis();
                break;
            }
        }
        if (msg == null) {
            if (!super.canRunTask()) {
                return null;
            }
            PubTask pt = taskQueue.poll();
            if (pt == null) {
                return null;
            }
            super.registerRunTask();
            if (pt.logId == null) {
                pt.logId = idGen.genId() + "";
            }
            groupLogId = pt.logId;
            runingMap.put(pt.logId, pt);
            msg = pt.list.get(pt.index);
            pt.writeTime = System.currentTimeMillis();
            pt.index++;
        }
        WriteTask task = new WriteTask();
        task.setLogId(groupLogId + "." + msg.getLogId());
        task.setBuf(packMessage(msg, task.getLogId(), messageId, sessionId, topicName));
        task.setSessionMessageId(messageId);
        return task;
    }

    @Override
    public void handleShutDown(SessionIdProvider sessionIdProvider) {
        fastFailedRunning(sessionIdProvider);
        //
        while (taskQueue.peek() != null) {
            PubTask pt = taskQueue.poll();
            pt.futrue.trigger(pt.list, "shut down", sessionIdProvider.getSessionId(false));
        }
    }

    @Override
    public void fastFailed(SessionIdProvider sessionIdProvider) {
        fastFailedRunning(sessionIdProvider);
    }

    @Override
    protected void fastFailedRunning(SessionIdProvider sessionIdProvider) {
        int needRelease = 0;
        for (String logId : runingMap.keySet()) {
            PubTask pt = runingMap.get(logId);
            List<Message> failedList = collectFailedTaskFromTask(pt);
            needRelease += pt.list.size();
            pt.futrue.trigger(failedList, "io error.", sessionIdProvider.getSessionId(false));
            super.unRegisterRunTask();
        }
        super.releaseToken(needRelease);
        runingMap.clear();
        // this.repeatTaskList.clear();

    }

    @Override
    protected void handleFinish(String logId, long status, SessionIdProvider sessionIdProvider) {
        String[] logIdArray = logId.split("\\.");
        String groupLogId = logIdArray[0];
        PubTask pt = runingMap.get(groupLogId);
        if (pt == null) {
            return;
        }
        pt.finList.add(logIdArray[1]);
        if (pt.finList.size() == pt.list.size()) {
            runingMap.remove(groupLogId);
            super.unRegisterRunTask();
            List<Message> failed = Collections.emptyList();
            pt.futrue.trigger(failed, null, sessionIdProvider.getSessionId(false));
        }
        super.releaseToken();
    }

    /**
     * 快速失败时收集所有未发送的消息.<br>
     * 
     * <p>
     * {@link PubTask#finList} 存储着已经发送完成的消息的logId，{@link PubTask#list} 存储着所有的
     * 本次需要发送的消息，如果一个{@link Message}的logiId没有出现在{@link PubTask#finList}中，则
     * 该消息没有被发送出现
     * </p>
     * 
     * 
     * @param pt {@link PubTask}
     * @return 还没有发送出去的消息
     */
    private List<Message> collectFailedTaskFromTask(PubTask pt) {
        List<Message> list = new LinkedList<Message>();
        for (Message m : pt.list) {
            if (pt.finList.indexOf(m.getLogId()) < 0) {
                list.add(m);
            }
        }
        return list;
    }

    /**
     * 按照bigpipe协议打包消息
     * 
     * @param msg {@link Message}消息对象
     * @param logId 发送的logId
     * @param messageId 消息Id，bigpipe使用messageId来做到不重不丢
     * @param sessionId sessionId
     * @param topicName bigpipe topic name
     * @return 消息打包buffer
     */
    private ByteBuffer packMessage(Message msg, String logId, long messageId, String sessionId, String topicName) {
        BigpipeCommand.Builder cmd = BigpipeCommand.newBuilder();
        cmd.setType(BigpipeCommand.CommandType.BMQ_SEND);
        MessageCommand.Builder msgCmd = cmd.getMessageBuilder();
        msgCmd.setDestination(topicName);
        msgCmd.setNoDup(false);
        msgCmd.setSessionMessageId(messageId);
        msgCmd.setReceiptId(logId);
        msgCmd.setSessionId(sessionId);

        if (!msg.packedMessage) {
            msgCmd.setMessageLength(8 + msg.getBody().length);
        } else {
            msgCmd.setMessageLength(msg.getBody().length);
        }
        byte[] cmdBuff = cmd.build().toByteArray();

        ByteBuffer bbf = ByteBuffer.allocate(NSHead.SIZE + 4 + cmdBuff.length + (int) msgCmd.getMessageLength());
        bbf.order(ByteOrder.LITTLE_ENDIAN);
        bbf.position(NSHead.SIZE);
        bbf.putInt(cmdBuff.length);
        bbf.put(cmdBuff);
        if (!msg.packedMessage) {
            bbf.putInt((int) msgCmd.getMessageLength());
            bbf.putInt(msg.getBody().length);
            bbf.put(msg.getBody());
        } else {
            bbf.put(msg.getBody());
        }

        NSHead header = NSHead.factory(getProvider());
        header.setBodyLen(cmdBuff.length + 4 + (int) msgCmd.getMessageLength());
        Adler32 checksum = new Adler32();
        byte[] msgBuf = bbf.array();
        checksum.update(msgBuf, NSHead.SIZE, (int) header.getBodyLen());
        long checkValue = (checksum.getValue() & 0xffffffff);
        header.setReserved((int) checkValue);
        System.arraycopy(header.toBytes(), 0, msgBuf, 0, NSHead.SIZE);

        bbf.flip();

        return bbf;
    }

    @Override
    public int getCurrentTaskCount() {
        return runingMap.size();
    }

    @Override
    public boolean handlePubTimeout(SessionIdProvider sessionIdProvider) {
        boolean hasTimeout = false;
        long topTime = System.currentTimeMillis() - super.socketConf.getIoTimeout();
        hasTimeout = hasTimeout || handleWaitingTimeout(topTime, sessionIdProvider);
        hasTimeout = hasTimeout || handleRunningingTimeout(topTime, sessionIdProvider);
        return false;
    }

    /**
     * 控制是否有消息没有开始处理，但已经超时
     * 
     * @param topTime 超时时间点
     * @param sessionIdProvider SessionIdProvider
     * @return 是否有超时
     */
    private boolean handleWaitingTimeout(long topTime, SessionIdProvider sessionIdProvider) {
        boolean hasTimeout = false;
        PubTask pt = null;

        while (null != (pt = this.taskQueue.peek())) {
            if (pt.pubTime <= topTime) {
                hasTimeout = true;
                this.taskQueue.poll();
                int needRelease = pt.list.size();
                pt.futrue.trigger(pt.list, "timeout, waiting to send", sessionIdProvider.getSessionId(false));
                super.releaseToken(needRelease);
            } else {
                break;
            }
        }

        return hasTimeout;
    }

    /**
     * 控制是否有消息没有处理完成，但已经超时
     * 
     * @param topTime 超时时间点
     * @param sessionIdProvider SessionIdProvider
     * @return 是否有已经开始发送，但超过超时时间后还没有发送完成的消息
     */
    private boolean handleRunningingTimeout(long topTime, SessionIdProvider sessionIdProvider) {
        boolean hasTimeout = false;
        List<String> removeList = new LinkedList<String>();
        for (String logId : runingMap.keySet()) {
            PubTask pt = runingMap.get(logId);
            if (pt.writeTime <= topTime) {
                hasTimeout = true;
                List<Message> failedList = collectFailedTaskFromTask(pt);
                int needRelease = pt.list.size();
                pt.futrue.trigger(failedList, "timeout,sending,logid is " + logId,
                        sessionIdProvider.getSessionId(false));
                super.releaseToken(needRelease);
                removeList.add(logId);
                // repeatTaskList.remove(pt);
            }
        }
        for (String logId : removeList) {
            super.unRegisterRunTask();
            this.runingMap.remove(logId);
        }
        return hasTimeout;
    }
}
