package com.baidu.bigpipe.transport.pub;

import java.nio.ByteBuffer;
import java.util.List;

import com.baidu.bigpipe.protocol.LogIdGen;
import com.baidu.bigpipe.protocol.SessionIdProvider;
import com.baidu.bigpipe.transport.TransportStrategy;
import com.baidu.bigpipe.transport.pub.context.WriteTask;

/**
 * publish消息的抽象策略,用来控制如何接收和发送消息,比如单个消息发送，批量消息发送和打包发送
 * 
 * @author hexiufeng
 * 
 */
public interface PublishStrategy extends TransportStrategy {
    /**
     * 提交要发送的单个消息
     * 
     * @param msg 消息对象
     */
    void submitMessage(Message msg);

    /**
     * 提交要发送的批量消息
     * 
     * @param msgList 消息list
     * @param futrue 内部futrue，用于触发任务结束事件
     */
    void submitMessage(List<Message> msgList, InternalFutrue futrue);

    /**
     * 生成下一个需要发送的任务，每一个任务可能需要发送单个、一批或者一包消息
     * 
     * @param idGen id生成器
     * @param messageId 消息id
     * @param sessionId sessionid
     * @param topicName bigpipe topic name
     * @return 任务
     */
    WriteTask getNextTask(LogIdGen idGen, long messageId, String sessionId, String topicName);

    /**
     * 当Session关闭时如何处理未发送的消息
     */
    /**
     * 当Session关闭时如何处理未发送的消息
     * 
     * @param sessionIdProvider sessionIdProvider
     */
    void handleShutDown(SessionIdProvider sessionIdProvider);

    /**
     * 一个发送任务被成功发送后需要处理的行为
     * 
     * @param buf 内容
     * @param sessionIdProvider SessionIdProvider
     */
    void finishPub(ByteBuffer buf, SessionIdProvider sessionIdProvider);

    /**
     * 发送任务时失败后，如何快速的失败
     * 
     * @param sessionIdProvider SessionIdProvider
     */
    void fastFailed(SessionIdProvider sessionIdProvider);

    /**
     * publish消息时由于内存queue容量是有限的,需要获取一个许可
     * 
     * @throws InterruptedException 线程被打断
     */
    void accqireToken() throws InterruptedException;

    /**
     * 消息处理完后需要释放一个许可
     */
    void releaseToken();

    /**
     * publish消息时由于内存queue容量是有限的,需要获取n个许可
     * 
     * @param n 获取token的数量
     * @throws InterruptedException 线程被打断异常
     */
    void accqireToken(int n) throws InterruptedException;

    /**
     * 消息处理完后需要释放n个许可
     * 
     * @param n 许可的数量
     */
    void releaseToken(int n);

    /**
     * 在tcp的双工模式下，可以并发的发送任务和接收信息，并发会消耗资源，使用本方法获取可并发处理的任务量
     * 
     * @return 并发任务数量
     */
    int getCurrentTaskCount();

    /**
     * 监控当前是否有任务已经持续很久，可能超时
     * 
     * @param sessionIdProvider SessionIdProvider
     * 
     * @return if need to reconnect tcp
     */
    boolean handlePubTimeout(SessionIdProvider sessionIdProvider);

}
