/*
 * Copyright (C) 2016 Baidu, Inc. All Rights Reserved.
 */
package com.baidu.bigpipe.transport.queue;

import java.io.IOException;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.zookeeper.KeeperException;
import org.slf4j.LoggerFactory;

import com.baidu.bigpipe.driver.converter.sub.MessageBodyConverter;
import com.baidu.bigpipe.protocol.McpackCommand;
import com.baidu.bigpipe.protocol.QueueAck;
import com.baidu.bigpipe.protocol.QueueMessage;
import com.baidu.bigpipe.protocol.QueueRequest;
import com.baidu.bigpipe.protocol.meta.NameService;
import com.baidu.bigpipe.protocol.meta.concept.InetAddress;
import com.baidu.bigpipe.protocol.meta.concept.QueueAddress;
import com.baidu.bigpipe.protocol.meta.exp.NameResolveException;
import com.baidu.bigpipe.protocol.meta.exp.QueueLocateException;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol;
import com.baidu.bigpipe.transport.AbstractSessionSocketStream;
import com.baidu.bigpipe.transport.BigpipeSessionSupport;
import com.baidu.bigpipe.transport.NHeadTransportStrategy;
import com.baidu.bigpipe.transport.Receiver;
import com.baidu.bigpipe.transport.SessionSocketStream;
import com.baidu.bigpipe.transport.TransportStrategy;
import com.baidu.bigpipe.transport.conf.BigPipeConf;
import com.baidu.bigpipe.transport.conf.SocketConf;
import com.baidu.bigpipe.transport.sub.PipeletIdAwareBigpipeMessageListener;
import com.baidu.mcpack.McpackException;

/**
 * 使用bio实现的queue订阅器
 * <p/>
 * 使用nio实现时很难控制超时，当网络环境不好时会发生丢包现象，这将导致socket无法读取数据，但从程序 的角度来看socket又是健康的，socket会被永远堵塞导致线程不再工作。在uc client和fountain-das中都
 * 遇到过这种问题，解决方案是设置读取超时，比如2分钟，如果超时则重新建立socket连接. 请使用bigPipeConf设置读取超时。
 *
 * @author liangxiaojie
 */
public class AsynchronousQueueSubscriberBioImpl extends BigpipeSessionSupport implements AsynchronousQueueSubscriber {
    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(AsynchronousQueueSubscriberBioImpl.class);

    private volatile Socket socket;
    /**
     * 传输策略，缺省是NsHead协议实现
     */
    private TransportStrategy transStrategy = new NHeadTransportStrategy();
    /**
     * bigpipe queuesrv的消息处理对象
     */
    private PipeletIdAwareBigpipeMessageListener queueListener;
    /**
     * 消息体转化器，可以将byte数组的消息体转化为java对象
     */
    private MessageBodyConverter bodyConverter;
    /**
     * bigpipe的静态属性配置
     */
    private BigPipeConf bigPipeConf;
    /**
     * 当前的订阅点，会被多线程使用，需要保证线程安全
     */
    private volatile long startPoint;

    public Receiver getReciever() {
        return reciever;
    }

    public void setReciever(Receiver reciever) {
        this.reciever = reciever;
    }

    public TransportStrategy getTransStrategy() {
        return transStrategy;
    }

    public void setTransStrategy(TransportStrategy transStrategy) {
        this.transStrategy = transStrategy;
    }

    public PipeletIdAwareBigpipeMessageListener getMessageListener() {
        return queueListener;
    }

    public void setQueueListener(PipeletIdAwareBigpipeMessageListener messageListener) {
        this.queueListener = messageListener;
    }

    public MessageBodyConverter getBodyConverter() {
        return bodyConverter;
    }

    public void setBodyConverter(MessageBodyConverter bodyConverter) {
        this.bodyConverter = bodyConverter;
    }

    public BigPipeConf getBigPipeConf() {
        return bigPipeConf;
    }

    public void setBigPipeConf(BigPipeConf bigPipeConf) {
        this.bigPipeConf = bigPipeConf;
    }

    @Override
    public void startSubscribe() {
        // this.queueListener = listener;
        setType(PIPE_TYPE.QUEUE.ordinal()); // 这是一个queue
        this.init(bigPipeConf);
    }

    @Override
    public void shutDown() {
        lifeController.setShutDown(true);
        try {
            safeCloseTcpConnect();
            lifeController.getShutDownWait().await(bigPipeConf.getShutDownTimeout(), TimeUnit.MINUTES);
        } catch (InterruptedException e) {
            // warn
            LOGGER.warn("shutdown error, ignore.", e);
        }

    }

    @Override
    protected void continueConfig(BigPipeConf conf) {
        this.bigPipeConf = conf;
        // queue订阅 不用配置 水位
    }

    /**
     * 订阅数据的线程
     */
    private void threadCoreFunction() {
        this.lifeController.getThreadRunning().countDown();
        while (true) {
            if (lifeController.isShutDown()) {
                lifeController.getShutDownWait().countDown();
                break;
            }
            if (socket == null) {
                buildConnect(false, bigPipeConf);
                if (lifeController.isShutDown()) {
                    lifeController.getShutDownWait().countDown();
                    break;
                }
            }
            // read data
            doSubscribeMessage();
        }
    }

    /**
     * 处理订阅消息数据，包含两个业务逻辑：
     * <ul>
     * <li>堵塞模式读取数据</li>
     * <li>解析数据并回调处理方法</li>
     * </ul>
     */
    private void doSubscribeMessage() {
        // read data
        try {
            ByteBuffer buffer = reciever.blockRecieve(socket);
            if (buffer == null) {
                LOGGER.info("read data from bigpipe failed [Is token right from email sent from system], resubscribe.");
                safeCloseTcpConnect();
                return;
            }
            QueueMessage message = McpackCommand.bufferToProtocol(buffer, QueueMessage.class);

            ByteBuffer body = ByteBuffer.wrap(message.getMsg_body());
            List<Object> mList = extract(body);
            // 正常运行处理不报错,那么就可以ack了
            queueListener.handle(mList, message.getPipelet_id(), message.getPipelet_msg_id());

            // ack to server
            sendAcknowledgement(message, false);

            // contine subscribe.. // 不需要发送订阅指令
            // continueNextMessage();

        } catch (SocketTimeoutException e) {
            // due to always log Timeout exception if no message arrive, change log level to warning
            LOGGER.warn(
                    "read timeout from bigpipe failed, resubscribe. It may be no new message arrives for a period "
                            + "time.");
            safeCloseTcpConnect();
        } catch (IOException e) {
            LOGGER.error("read from bigpipe failed,unkown io error, resubscribe.", e);
            safeCloseTcpConnect();
        } catch (RuntimeException e) {
            LOGGER.error("read from bigpipe failed,unkown error, resubscribe.", e);
            safeCloseTcpConnect();
        } catch (Exception e) {
            LOGGER.error("read from bigpipe failed,unkown error, resubscribe.", e);
            safeCloseTcpConnect();
        }
    }

    /**
     * <p>一般不建议直接使用此方法，可以使用acknowlegeXXX的方法</p>
     * <p>用于向queuesvr回复指定消息的ack报文，服务端将移除此消息（消费掉）</p>
     *
     * @param message 需要确认消费的消息对象
     * @param isFake  如果为true，则服务端接下来将进行fake行为（停止推送）
     *
     * @return 发给服务端的ack协议报文对象
     *
     * @throws IOException
     */
    public McpackCommand sendAcknowledgement(QueueMessage message, boolean isFake)
            throws IOException {
        QueueAck ack = new QueueAck();
        if (isFake) {
            ack.setCmd_no(McpackCommand.ACK_QUEUE_FAKE_DATA);
        } else {
            ack.setCmd_no(McpackCommand.ACK_QUEUE_TRUE_DATA);
        }
        ack.setQueue_name(bigPipeConf.getQueue());
        ack.setPipe_name(message.getPipe_name());
        ack.setPipelet_id(message.getPipelet_id());
        ack.setPipelet_msg_id(message.getPipelet_msg_id());
        ack.setSeq_id(message.getSeq_id());

        try {
            ByteBuffer buffer = McpackCommand.protocolToBuffer(ack);
            sendByteBufferData2Socket(socket, buffer);
        } catch (IOException e) {
            throw e;
        } catch (McpackException e) {
            IOException ex = new IOException(
                    "McpackException happend during request" + e.getMessage());
            ex.initCause(e);
            throw ex;
        }
        return ack;
    }

    /**
     * 发送继续订阅消息的指令
     *
     * @param messageComand 订阅指令
     */
    private void continueNextMessage(BigpipePBProtocol.MessageCommand messageComand) {
        BigpipePBProtocol.BigpipeCommand cmd = buildResponseMessage(messageComand);
        ByteBuffer buff = this.transStrategy.buildSimpleCommand(cmd.toByteArray());
        try {
            sendByteBufferData2Socket(socket, buff);
        } catch (IOException e) {
            LOGGER.error("write to bigpipe failed,unkown error, resubscribe.", e);
            safeCloseTcpConnect();
        }
    }

    /**
     * 构建继续订阅消息的指令
     *
     * @param messageComand 当前消息
     *
     * @return bigpipe继续订阅消息指令
     */
    private BigpipePBProtocol.BigpipeCommand buildResponseMessage(BigpipePBProtocol.MessageCommand messageComand) {
        BigpipePBProtocol.BigpipeCommand.Builder cmd = BigpipePBProtocol.BigpipeCommand.newBuilder();
        cmd.setType(BigpipePBProtocol.BigpipeCommand.CommandType.BMQ_ACK);
        BigpipePBProtocol.AckCommand.Builder ackCmd = cmd.getAckBuilder();
        ackCmd.setDestination(messageComand.getDestination());
        ackCmd.setAckType(1);
        ackCmd.setTopicMessageId(messageComand.getTopicMessageId());
        ackCmd.setReceiptId(messageComand.getReceiptId());
        BigpipePBProtocol.BigpipeCommand command = cmd.build();
        return command;
    }

    /**
     * 解析并抽取消息数据
     *
     * @param buf byte[] 数据
     *
     * @return 抽取出的消息
     */
    private List<Object> extract(ByteBuffer buf) {
        List<Object> list = new LinkedList<Object>();
        buf.order(ByteOrder.LITTLE_ENDIAN);
        // int allCount = buf.remaining();
        int size = buf.getInt();
        LOGGER.info("body size:" + size);
        // ignore this check, because some bigpipe client don't Comply with
        // bigpipe package protocol.
        // if (size != allCount) {
        // return list;
        // }
        // int msgSize = 0;
        int count = 0;
        while (buf.remaining() > 4) {
            int bodySize = buf.getInt();

            // fix here if some body give a error body size with buf.getInt
            if (bodySize > buf.remaining()) {
                bodySize = buf.remaining();
            }

            byte[] array = new byte[bodySize];
            buf.get(array);
            // msgSize++;
            list.add(bodyConverter.bin2Object(array));
            count++;
        }
        return list;
    }

    /**
     * 开启selector.select线程
     */
    public void start(BigPipeConf conf) {
        // 在监控线程启动前首先构建tcp channel,监控线程启动后会检测到该channel,后续由监控线程来
        // 重新构建channel，此过程即使tcp channel不是volatile修饰的也能保证线程安全
        buildConnect(true, conf);

        Thread t = new Thread(new Runnable() {

            @Override
            public void run() {
                threadCoreFunction();
            }

        });
        String threadName = "bigpipe-client-thread-";
        if (conf.getThreadName() != null) {
            threadName = threadName + conf.getThreadName() + "-";
        }
        t.setName(threadName + t.getId());
        t.start();
        try {
            lifeController.getThreadRunning().await();
        } catch (InterruptedException e) {
            LOGGER.error("start thread error.", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 由被ensureStartPoint转化过的point转换成zk position
     *
     * @param point 被ensureStartPoint转化过得startpoint
     *
     * @return zk position
     */
    private long convertPositionFromFixedStartPoint(long point) {
        if (point == BigPipeConf.COUNT_MAX) {
            return Long.MAX_VALUE;
        }
        return point;
    }

    /**
     * 连接bigpipe失败时线程等待
     *
     * @param cnt 已经重试的次数
     */
    @Override
    protected void waitingForConnect(int cnt) {
        LOGGER.warn("reconnect bigpipe failed.{} times", cnt);
        long sleep = 60000;
        if (cnt < 5) {
            sleep = 500;
        }
        if (cnt > 5) {
            LOGGER.warn("WARNING:failed to reconnect bigpipe more than 60s.");
        }
        try {
            Thread.sleep(sleep);
        } catch (InterruptedException e) {
            // ignore
        }
    }

    /**
     * 构建并发送订阅指令
     *
     * @param socket   socket
     * @param reciever 数据接收器
     *
     * @throws IOException io异常
     */
    private void startSubcribe(Socket socket, Receiver reciever) throws IOException {

        // 开始发起REQ_QUEUE_DATA命令,请求数据
        QueueRequest request = new QueueRequest();
        request.setCmd_no(McpackCommand.REQ_QUEUE_DATA);
        request.setQueue_name(bigPipeConf.getQueue());
        request.setToken(bigPipeConf.getToken4Queue());
        request.setWindow_size(bigPipeConf.getWindowSize());
        request.setApi_version("driver4j");
        try {
            ByteBuffer buf = McpackCommand.protocolToBuffer(request);
            socket.getOutputStream().write(buf.array(), 0, buf.limit());
        } catch (IOException e) {
            throw e;
        } catch (McpackException e) {
            IOException ex = new IOException(
                    "McpackException happend during request" + e.getMessage());
            ex.initCause(e);
            throw ex;
        }
    }

    /**
     * 写bytebuffer数据到socket
     *
     * @param socket socket
     * @param buf    bytebuffer
     *
     * @throws IOException io异常构建并发送订阅指令
     */
    private void sendByteBufferData2Socket(Socket socket, ByteBuffer buf) throws IOException {
        socket.getOutputStream().write(buf.array(), 0, buf.limit());
    }

    /**
     * 这个openStream是打开关于queue的连接地址
     *
     * @param addr {@link InetAddress}地址
     * @param conf {@link SocketConf}网络相关配置
     *
     * @return
     *
     * @throws IOException
     */
    @Override
    protected SessionSocketStream openStream(InetAddress addr, SocketConf conf) throws IOException {
        socket = new Socket();
        socket.connect(addr.getAddress(), conf.getConectTimeout());
        socket.setSoTimeout(conf.getIoTimeout());

        return new AbstractSessionSocketStream() {

            @Override
            protected Socket buildSocketIfNotExist() {
                return socket;
            }

            @Override
            protected TransportStrategy getTransStrategy() {
                return transStrategy;
            }

            @Override
            protected Receiver getReceiver() {
                return reciever;
            }

            @Override
            protected int getRole() {
                return BIGPIPE_ROLE_SUB;
            }

            @Override
            protected void afterCreateSession() throws IOException {
                startSubcribe(socket, reciever);
            }

        };
    }

    @Override
    protected InetAddress lookupAddr(NameService ns, String queueName) throws NameResolveException,
            KeeperException, QueueLocateException {
        QueueAddress queueAddress = ns.lookupQueue(queueName);
        return queueAddress;
    }

    @Override
    protected void handleFastFailed(boolean needBlocking) {
        // donothing
    }

    /**
     * 关闭socket
     */
    @Override
    protected void safeCloseTcpConnect() {
        if (socket == null) {
            return;
        }
        try {
            socket.close();
        } catch (IOException e) {
            LOGGER.info("safeCloseTcpConnect", e);
        } finally {
            socket = null;
        }
    }

}
