package com.baidu.bigpipe.transport.sub;

import java.io.IOException;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.baidu.bigpipe.driver.converter.sub.MessageBodyConverter;
import com.baidu.bigpipe.position.store.SubcribePositionStore;
import com.baidu.bigpipe.protocol.BigpipePacket;
import com.baidu.bigpipe.protocol.meta.NameService;
import com.baidu.bigpipe.protocol.meta.concept.InetAddress;
import com.baidu.bigpipe.protocol.meta.concept.TopicAddress;
import com.baidu.bigpipe.protocol.meta.exp.NameResolveException;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.AckCommand;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.BigpipeCommand;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.BigpipeCommand.CommandType;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.MessageCommand;
import com.baidu.bigpipe.protocol.pb.BigpipePBProtocol.SubscribeCommand;
import com.baidu.bigpipe.transport.AbstractSessionSocketStream;
import com.baidu.bigpipe.transport.BigpipeSessionSupport;
import com.baidu.bigpipe.transport.NHeadTransportStrategy;
import com.baidu.bigpipe.transport.Receiver;
import com.baidu.bigpipe.transport.SessionSocketStream;
import com.baidu.bigpipe.transport.TransportStrategy;
import com.baidu.bigpipe.transport.conf.BigPipeConf;
import com.baidu.bigpipe.transport.conf.SocketConf;
import com.google.protobuf.InvalidProtocolBufferException;

/**
 * 使用bio实现的订阅器
 * <p>
 * 使用nio实现时很难控制超时，当网络环境不好时会发生丢包现象，这将导致socket无法读取数据，但从程序 的角度来看socket又是健康的，socket会被永远堵塞导致线程不再工作。在uc client和fountain-das中都
 * 遇到过这种问题，解决方案是设置读取超时，比如2分钟，如果超时则重新建立socket连接. 请使用bigPipeConf设置读取超时。
 * </p>
 * <p/>
 * changelog by xiemalin
 * 1. fix {@link #extract(ByteBuffer, long[], MessageCommand[])} method should check  bodySize <= buf.remaining()
 *
 * @author hexiufeng
 * @author xiemalin
 */
public class AsynchronousSubscriberBioImpl extends BigpipeSessionSupport implements AsynchronousSubscriber {
    private static final Logger LOGGER = LoggerFactory.getLogger(AsynchronousSubscriberBioImpl.class);

    private volatile Socket socket;
    /**
     * 传输策略，缺省是NsHead协议实现
     */
    private TransportStrategy transStrategy = new NHeadTransportStrategy();
    /**
     * bigpipe的消息处理对象
     */
    private BigpipeMessageListener messageListener;
    /**
     * 消息体转化器，可以将byte数组的消息体转化为java对象
     */
    private MessageBodyConverter bodyConverter;
    /**
     * 已处理完成消息的的订阅点
     */
    private SubcribePositionStore positionStore;
    /**
     * bigpipe的静态属性配置
     */
    private BigPipeConf bigPipeConf;
    /**
     * 当前的订阅点，会被多线程使用，需要保证线程安全
     */
    private volatile long startPoint;

    public Receiver getReciever() {
        return reciever;
    }

    public void setReciever(Receiver reciever) {
        this.reciever = reciever;
    }

    public TransportStrategy getTransStrategy() {
        return transStrategy;
    }

    public void setTransStrategy(TransportStrategy transStrategy) {
        this.transStrategy = transStrategy;
    }

    public BigpipeMessageListener getMessageListener() {
        return messageListener;
    }

    public void setMessageListener(BigpipeMessageListener messageListener) {
        this.messageListener = messageListener;
    }

    public MessageBodyConverter getBodyConverter() {
        return bodyConverter;
    }

    public void setBodyConverter(MessageBodyConverter bodyConverter) {
        this.bodyConverter = bodyConverter;
    }

    public SubcribePositionStore getPositionStore() {
        return positionStore;
    }

    public void setPositionStore(SubcribePositionStore positionStore) {
        this.positionStore = positionStore;
    }

    @Override
    public void startSubscribe(BigpipeMessageListener listener, BigPipeConf conf) {
        messageListener = listener;
        this.init(conf);
    }

    @Override
    public void shutDown() {
        lifeController.setShutDown(true);
        try {
            safeCloseTcpConnect();
            lifeController.getShutDownWait().await(bigPipeConf.getShutDownTimeout(), TimeUnit.MINUTES);
        } catch (InterruptedException e) {
            // ignore
        }

    }

    @Override
    protected void continueConfig(BigPipeConf conf) {
        this.bigPipeConf = conf;
        try {
            ensureStartPoint(initStartPoint(conf));
        } catch (NameResolveException e) {
            throw new RuntimeException(e);
        } catch (KeeperException ke) {
            throw new RuntimeException(ke);
        }
    }

    /**
     * 订阅数据的线程
     */
    private void threadCoreFunction() {
        this.lifeController.getThreadRunning().countDown();
        while (true) {
            if (lifeController.isShutDown()) {
                lifeController.getShutDownWait().countDown();
                break;
            }
            if (socket == null) {
                buildConnect(false, bigPipeConf);
                if (lifeController.isShutDown()) {
                    lifeController.getShutDownWait().countDown();
                    break;
                }
            }
            // read data
            doSubscribeMessage();
        }
    }

    /**
     * 处理订阅消息数据，包含两个业务逻辑：
     * <ul>
     * <li>堵塞模式读取数据</li>
     * <li>解析数据并回调处理方法</li>
     * </ul>
     */
    private void doSubscribeMessage() {
        // read data
        try {
            ByteBuffer buffer = reciever.blockRecieve(socket);
            if (buffer == null) {
                LOGGER.info("read data from bigpipe failed, resubscribe.");
                safeCloseTcpConnect();
                return;
            }
            // 回调并处理
            long[] ptHolder = {0};
            MessageCommand[] msgHolder = {null};
            List<Object> mList = extract(buffer, ptHolder, msgHolder);
            if (msgHolder[0] == null) {
                safeCloseTcpConnect();
                return;
            }
            if (mList.size() > 0) {
                messageListener.handle(mList, ptHolder[0]);
                this.startPoint = ptHolder[0] + 1;
            } else {
                LOGGER.info("recieve null message.");
            }
            // contine subscribe
            continueNextMessage(msgHolder[0]);

        } catch (SocketTimeoutException e) {
            // due to always log Timeout exception if no message arrive, change log level to warning
            LOGGER.warn(
                    "read timeout from bigpipe failed, resubscribe. It may be no new message arrives for a period "
                            + "time.");
            safeCloseTcpConnect();
        } catch (IOException e) {
            LOGGER.error("read from bigpipe failed,unkown io error, resubscribe.", e);
            safeCloseTcpConnect();
        } catch (RuntimeException e) {
            LOGGER.error("read from bigpipe failed,unkown error, resubscribe.", e);
            safeCloseTcpConnect();
        } catch (Exception e) {
            LOGGER.error("read from bigpipe failed,unkown error, resubscribe.", e);
            safeCloseTcpConnect();
        }
    }

    /**
     * 发送继续订阅消息的指令
     *
     * @param messageComand 订阅指令
     */
    private void continueNextMessage(MessageCommand messageComand) {
        BigpipeCommand cmd = buildResponseMessage(messageComand);
        ByteBuffer buff = this.transStrategy.buildSimpleCommand(cmd.toByteArray());
        try {
            sendByteBufferData2Socket(socket, buff);
        } catch (IOException e) {
            LOGGER.error("write to bigpipe failed,unkown error, resubscribe.", e);
            safeCloseTcpConnect();
        }
    }

    /**
     * 构建继续订阅消息的指令
     *
     * @param messageComand 当前消息
     *
     * @return bigpipe继续订阅消息指令
     */
    private BigpipeCommand buildResponseMessage(MessageCommand messageComand) {
        BigpipeCommand.Builder cmd = BigpipeCommand.newBuilder();
        cmd.setType(BigpipeCommand.CommandType.BMQ_ACK);
        AckCommand.Builder ackCmd = cmd.getAckBuilder();
        ackCmd.setDestination(messageComand.getDestination());
        ackCmd.setAckType(1);
        ackCmd.setTopicMessageId(messageComand.getTopicMessageId());
        ackCmd.setReceiptId(messageComand.getReceiptId());
        BigpipeCommand command = cmd.build();
        return command;
    }

    /**
     * 解析并抽取消息数据
     *
     * @param buf         byte[] 数据
     * @param pointHolder 记录点的占位符
     * @param msgHolder   消息命令的占位符
     *
     * @return 抽取出的消息
     */
    private List<Object> extract(ByteBuffer buf, long[] pointHolder, MessageCommand[] msgHolder) {
        List<Object> list = new LinkedList<Object>();
        buf.order(ByteOrder.LITTLE_ENDIAN);

        byte[] cmdbytes = new byte[buf.getInt()];
        buf.get(cmdbytes);
        BigpipeCommand cmd = null;
        try {
            cmd = BigpipeCommand.parseFrom(cmdbytes);
        } catch (InvalidProtocolBufferException e) {
            LOGGER.error("BigpipeCommand.parseFrom error.", e);
            return list;
        }
        msgHolder[0] = cmd.getMessage();
        if (cmd.getType() != BigpipeCommand.CommandType.BMQ_MESSAGE) {
            // output error
            LOGGER.error(cmd.getError().toString());
            return list;
        }

        pointHolder[0] = cmd.getMessage().getTopicMessageId();
        // int allCount = buf.remaining();
        int size = buf.getInt();
        LOGGER.info("body size:" + size);
        // ignore this check, because some bigpipe client don't Comply with
        // bigpipe package protocol.
        // if (size != allCount) {
        // return list;
        // }
        // int msgSize = 0;
        int count = 0;
        while (buf.remaining() > 4) {
            int bodySize = buf.getInt();

            // fix here if some body give a error body size with buf.getInt
            if (bodySize > buf.remaining()) {
                bodySize = buf.remaining();
            }

            byte[] array = new byte[bodySize];
            buf.get(array);
            // msgSize++;
            list.add(bodyConverter.bin2Object(array));
            count++;
        }
        return list;
    }

    /**
     * 开启selector.select线程
     */
    public void start(BigPipeConf conf) {
        // 在监控线程启动前首先构建tcp channel,监控线程启动后会检测到该channel,后续由监控线程来
        // 重新构建channel，此过程即使tcp channel不是volatile修饰的也能保证线程安全
        buildConnect(true, conf);

        Thread t = new Thread(new Runnable() {

            @Override
            public void run() {
                threadCoreFunction();
            }

        });
        String threadName = "bigpipe-client-thread-";
        if (conf.getThreadName() != null) {
            threadName = threadName + conf.getThreadName() + "-";
        }
        t.setName(threadName + t.getId());
        t.start();
        try {
            lifeController.getThreadRunning().await();
        } catch (InterruptedException e) {
            LOGGER.error("start thread error.", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 由被ensureStartPoint转化过的point转换成zk position
     *
     * @param point 被ensureStartPoint转化过得startpoint
     *
     * @return zk position
     */
    private long convertPositionFromFixedStartPoint(long point) {
        if (point == BigPipeConf.COUNT_MAX) {
            return Long.MAX_VALUE;
        }
        return point;
    }

    /**
     * 连接bigpipe失败时线程等待
     *
     * @param cnt 已经重试的次数
     */
    @Override
    protected void waitingForConnect(int cnt) {
        LOGGER.warn("reconnect bigpipe failed.{} times", cnt);
        long sleep = 60000;
        if (cnt < 5) {
            sleep = 500;
        }
        if (cnt > 5) {
            LOGGER.warn("WARNING:failed to reconnect bigpipe more than 60s.");
        }
        try {
            Thread.sleep(sleep);
        } catch (InterruptedException e) {
            // ignore
        }
    }

    /**
     * 构建并发送订阅指令
     *
     * @param socket   socket
     * @param reciever 数据接收器
     *
     * @throws IOException io异常
     */
    private void startSubcribe(Socket socket, Receiver reciever) throws IOException {
        BigpipeCommand.Builder cmd = BigpipeCommand.newBuilder();
        cmd.setType(BigpipeCommand.CommandType.BMQ_SUBSCRIBE);
        SubscribeCommand.Builder subCmd = cmd.getSubscribeBuilder();
        subCmd.setDestination(pipeRuntime.getTopicName());
        subCmd.setStartPoint(this.startPoint);
        LOGGER.info("subscribe start point:" + startPoint);
        subCmd.setReceiptId(System.currentTimeMillis() + "-" + Math.random());
        BigpipeCommand command = cmd.build();
        ByteBuffer buf = transStrategy.buildSimpleCommand(command.toByteArray());
        sendByteBufferData2Socket(socket, buf);
        ByteBuffer read = reciever.blockRecieve(socket);
        if (read == null) {
            throw new RuntimeException("connect session failed");
        }
        BigpipePacket sub = AbstractSessionSocketStream.parseCommand(read);
        if (sub.getCommand().getType() != CommandType.BMQ_RECEIPT) {
            throw new RuntimeException(sub.getCommand().getError().toString());
        }
    }

    /**
     * 写bytebuffer数据到socket
     *
     * @param socket socket
     * @param buf    bytebuffer
     *
     * @throws IOException io异常构建并发送订阅指令
     */
    private void sendByteBufferData2Socket(Socket socket, ByteBuffer buf) throws IOException {
        socket.getOutputStream().write(buf.array(), 0, buf.limit());
    }

    /**
     * 初始化start point
     *
     * @param conf BigPipeConf
     *
     * @return 同步点
     */
    private long initStartPoint(BigPipeConf conf) {
        if (this.positionStore != null) {
            Long pt = this.positionStore.loadPosition();
            if (pt != null) {
                pt++;
                return pt;
            }
        }
        return conf.getDefStartPoint();
    }

    /**
     * 根据输入的start point计算出bigpipe真正的start point
     *
     * @param point 同步点
     *
     * @throws NameResolveException zk异常
     * @throws KeeperException      zk异常
     */
    private void ensureStartPoint(long point) throws NameResolveException, KeeperException {
        if (point == BigPipeConf.COULD_ZERO) {
            NameService ns = pipeRuntime.getNs();
            String pipeletName = getPipeletOrQueueName();
            TopicAddress address = ns.lookupForSub(pipeletName, 0);
            if (address == null) {
                throw new NameResolveException(pipeletName, Long.MAX_VALUE, "sub stripe not found");
            }
            this.startPoint = address.getStripe().getBeginPos();
            return;
        }
        if (point == BigPipeConf.COUNT_MAX) {
            this.startPoint = BigPipeConf.COUNT_MAX;
            return;
        }
        this.startPoint = point;
    }

    @Override
    protected SessionSocketStream openStream(InetAddress addr, SocketConf conf) throws IOException {
        socket = new Socket();
        socket.connect(addr.getAddress(), conf.getConectTimeout());
        socket.setSoTimeout(conf.getIoTimeout());

        return new AbstractSessionSocketStream() {

            @Override
            protected Socket buildSocketIfNotExist() {
                return socket;
            }

            @Override
            protected TransportStrategy getTransStrategy() {
                return transStrategy;
            }

            @Override
            protected Receiver getReceiver() {
                return reciever;
            }

            @Override
            protected int getRole() {
                return BIGPIPE_ROLE_SUB;
            }

            @Override
            protected void afterCreateSession() throws IOException {
                startSubcribe(socket, reciever);
            }

        };
    }

    //    @Override
    //    protected SessionSocketStream openStreamForQueue(QueueAddress addr, SocketConf conf)
    //            throws IOException {
    //        // do nothing
    //        return null;
    //    }

    @Override
    protected TopicAddress lookupAddr(NameService ns, String pipeletName) throws NameResolveException, KeeperException {
        return ns.lookupForSub(pipeletName, convertPositionFromFixedStartPoint(this.startPoint));
    }

    @Override
    protected void handleFastFailed(boolean needBlocking) {
        // donothing
    }

    /**
     * 关闭socket
     */
    @Override
    protected void safeCloseTcpConnect() {
        if (socket == null) {
            return;
        }
        try {
            socket.close();
        } catch (IOException e) {
            LOGGER.info("safeCloseTcpConnect", e);
        } finally {
            socket = null;
        }
    }
}
