/*
 * Copyright 2002-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.baidu.driver4j.bns;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Random;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import com.baidu.driver4j.bns.BNSException.ErrorCode;

/**
 * BNS query proxy which query from BNS agent supports.
 * 
 * @author xiemalin
 * @since 1.0.0
 */
public class BNSQueryAgentProxy implements InstanceQuery {

    /**
     * default instance status for {@link BNSQueryProxy} to filter 
     */
    private static final int QUERY_INSTANCE_STATUS = 0;

    /**
     * Split string for port name and port number
     */
    private static final String PORT_NAME_SPLIT = "=";

    /**
     * Split string for multiple port
     */
    private static final String MULTI_PORT_SPLIT = ",";

    /**
     * default token
     */
    private static final String DEFAULT_TOKEN = "00000000";

    /**
     * PROC_NET_ROUTE
     */
    private static final String PROC_NET_ROUTE_COMMAND = "/proc/net/route";

    /**
     * JPAAS_HOST
     */
    private static final String JPAAS_HOST_NAME = "JPAAS_HOST";

    /**
     * local IP starts mark
     */
    private static final String LOCAL_IP_STARTS = "192.168";

    /**
     * LOGGER log this class
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(BNSQueryAgentProxy.class.getName());

    private BNSQueryProxy bnsQueryProxy;

    private final static String LOCAL_NAMED_ADDR = "127.0.0.1";
    private final static int LOCAL_NAMED_PORT = 793;
    private final static String REMOTE_NAMED_ADDR = "unamed.noah.baidu.com";
    private final static int REMOTE_NAMED_PORT = 793;

    private final static int DEFAULT_TIMEOUT_MS = 3000;
    private final static short MSG_TYPE_NAMING_REQ = 1;
    private final static short MSG_TYPE_AUTH_REQ = 2;
    private final static short MSG_TYPE_NAMING_RESP = 3;
    private final static short MSG_TYPE_AUTH_RESP = 4;
    private final static short MSG_TYPE_CONF_REQ = 15;
    private final static short MSG_TYPE_CONF_RESP = 16;

    private final static int RET_CODE_SERVICE_NOTEXIST = -1;
    private final static int RET_CODE_BEYOND_THRESHOLD = -16;

    private final static int MIN_TIME_OUT = 500;

    private int connectTimeout = 1000;
    private int readTimeout = DEFAULT_TIMEOUT_MS;

    private BNSQueryAgentProxy(String bnsService) {
        bnsQueryProxy = BNSQueryProxy.proxy(bnsService);
    }
    
    private BNSQueryAgentProxy(BNSQueryProxy proxy) {
        Assert.notNull(proxy, "Property 'proxy' of type BNSQueryProxy is null ");
        this.bnsQueryProxy = proxy;
    }

    public static BNSQueryAgentProxy proxy() {
        return new BNSQueryAgentProxy(BNSQueryProxy.DEFAULT_BNS_SERVER);
    }
    
    public static BNSQueryAgentProxy proxy(String bnsServiceUrl) {
        return new BNSQueryAgentProxy(bnsServiceUrl);
    }
    
    public static BNSQueryAgentProxy proxy(BNSQueryProxy proxy) {
        return new BNSQueryAgentProxy(proxy);
    }

    /**
     * setter method for property connectTimeout
     * 
     * @param connectTimeout the connectTimeout to set
     */
    public void setConnectTimeout(int connectTimeout) {
        this.connectTimeout = connectTimeout;

        bnsQueryProxy.setConnectTimeout(connectTimeout);
    }

    /**
     * setter method for property readTimeout
     * 
     * @param readTimeout the readTimeout to set
     */
    public void setReadTimeout(int readTimeout) {
        this.readTimeout = readTimeout;

        bnsQueryProxy.setReadTimeout(readTimeout);
    }

    /**
     * Get only one instance by {@link SelectionStrategy}.
     * 
     * @param serviceName service name
     * @param strategy {@link SelectionStrategy}.
     * @return {@link Instance} object
     * @throws BNSException in case of access BNS server failed
     */
    public Instance getInstanceByService(final String serviceName, SelectionStrategy strategy) throws BNSException {
        return getInstanceByService(serviceName, strategy, readTimeout);
    }

    /**
     * Get only one instance by {@link SelectionStrategy}.
     * 
     * @param serviceName service name
     * @param strategy {@link SelectionStrategy}.
     * @param timeout time out on reading
     * @return {@link Instance} object
     * @throws BNSException in case of access BNS server failed
     */
    public Instance getInstanceByService(final String serviceName, SelectionStrategy strategy, int timeout)
            throws BNSException {
        if (strategy == null) {
            throw new BNSException(ErrorCode.BNS_INPUT_ERROR_PARAM, "SelectionStrategy 'strategy' is null.");
        }

        List<Instance> instances = getInstanceByService(serviceName, timeout);
        return selectOneByStrategy(instances, strategy);

    }

    /**
     * To select only one instance by {@link SelectionStrategy}. or return null if a empty instances list.
     * @param instances target instance list
     * @param strategy {@link SelectionStrategy} instance
     * @return {@link Instance} object
     */ 
    protected Instance selectOneByStrategy(List<Instance> instances, SelectionStrategy strategy) {
        if (instances == null || instances.isEmpty()) {
            return null;
        }

        Instance ret = null;
        int decideValue = -1;
        for (Instance instance : instances) {
            int deciding = strategy.deciding(instance);
            if (deciding < 0) {
                deciding = 0;
            } else if (deciding > 100) {
                deciding = SelectionStrategy.SELECTED;
            }
            // quick selected
            if (deciding == SelectionStrategy.SELECTED) {
                return instance;
            }

            if (deciding > decideValue) {
                ret = instance;
                decideValue = deciding;
            }
        }

        return ret;
    }

    /**
     * Get the instances list of given serviceName.
     * 
     * This function will throws BNSException with several kind of error code. Check BNSException.errorToString for more
     * information. <br/>
     * You shold take BNS_SERVICE_BEYOND_THRESHOLD exceptin seriously, In many times, just use the previous result.
     * 
     * @param serviceName
     * @return List<BNSInstance> instances list of the given serviceName
     * @throws BNSException in case of access BNS server failed
     * @see #getInstanceByService(String, int)
     */
    public List<Instance> getInstanceByService(final String serviceName) throws BNSException {
        return getInstanceByService(serviceName, readTimeout);

    }

    /**
     * Get the instances list of given serviceName. <br>
     * first it will query from local agent then try to query from remote agent, at last query from HTTP API
     * 
     * This function will throws BNSException with several kind of error code. Check BNSException.errorToString for more
     * information. <br/>
     * You should take BNS_SERVICE_BEYOND_THRESHOLD exception seriously, In many times, just use the previous result.
     * 
     * @param serviceName
     * @param timeoutMs minimum is 500ms
     * @return List<BNSInstance> instances list of the given serviceName
     * @throws BNSException in case of access BNS server failed
     */
    public List<Instance> getInstanceByService(final String serviceName, int timeoutMs) throws BNSException {
        if (StringUtils.isEmpty(serviceName)) {
            throw new BNSException(ErrorCode.BNS_INPUT_ERROR_PARAM, "serivce name is blank.");
        }

        int timeout = timeoutMs;
        if (timeoutMs < MIN_TIME_OUT) {
            timeout = MIN_TIME_OUT;
        }
        
        // for supports office network
        List<Instance> instances = bnsQueryProxy.getInstances(serviceName, QUERY_INSTANCE_STATUS);
        return instances;
        
    }

    /**
     * Get the {@link Service} object by noah service name.
     *
     * @param serviceName serivce name
     * @param timeoutMs minimum is 500ms
     * @return {@link Service} instance
     * @see #getService(String, int)
     */
    public Service getService(final String serviceName) {
        return getService(serviceName, readTimeout);
    }

    /**
     * Get the {@link Service} object first it will query from local agent then try to query from remote agent, at last
     * query from HTTP API
     *
     * @param serviceName serivce name
     * @param timeoutMs minimum is 500ms
     * @return {@link Service} instance
     */
    public Service getService(final String serviceName, int timeoutMs) {
        if (StringUtils.isEmpty(serviceName)) {
            throw new BNSException(ErrorCode.BNS_INPUT_ERROR_PARAM, "serivce name is blank.");
        }

        int timeout = timeoutMs;
        if (timeoutMs < MIN_TIME_OUT) {
            timeout = MIN_TIME_OUT;
        }
        
        // for supports office network
        Service service = bnsQueryProxy.getService(serviceName);
        return service;

    }

    private void checkResponseCode(int responseCode) {
        if (responseCode != 0) {
            String msg = "";
            ErrorCode errorCode;
            if (responseCode == RET_CODE_SERVICE_NOTEXIST) {
                errorCode = ErrorCode.BNS_SERVICE_NOTEXIST;
            } else if (responseCode == RET_CODE_BEYOND_THRESHOLD) {
                errorCode = ErrorCode.BNS_SERVICE_BEYOND_THRESHOLD;
            } else {
                errorCode = ErrorCode.BNS_RET_UNKNOWN;
                msg = "Unkow retCode from naming-agent:" + errorCode;
            }
            throw new BNSException(errorCode, msg);
        }
    }

    /**
     * Communicate with target host
     * 
     * @param host host name
     * @param port port number
     * @param requestBodyBytes request bytes
     * @param requestMsgType request type
     * @param expectResponseType response type
     * @param timeoutMs read time out
     * @return byte[] response content
     * @throws BNSException in case of access BNS service failed
     */
    protected byte[] doRequest(String host, int port, byte[] requestBodyBytes, short requestMsgType,
            short expectResponseType, int timeoutMs) throws BNSException {
        Socket socket = null;
        try {
            int timeout;
            if (timeoutMs > 0) {
                timeout = timeoutMs;
            } else {
                timeout = readTimeout;
            }

            socket = SocketUtils.createSocket(host, port, connectTimeout, timeout);
            socket.setReuseAddress(true);
            SocketUtils.writeSocket(socket, getBNSMsgBytes(requestBodyBytes, requestMsgType));
            InputStream is = socket.getInputStream();

            return readBNSMsg(is, expectResponseType);
        } catch (Exception e) {
            throw new BNSException(ErrorCode.BNS_TALK_TO_AGENT_ERROR, e.getMessage());
        } finally {
            try {
                if (SocketUtils.isSocketConnected(socket)) {
                    socket.close();
                }
            } catch (Exception e) {
            }
        }
    }

    protected byte[] getBNSMsgBytes(byte[] requestBodyBytes, short requestMsgType) throws IOException {
        BNSHead reqHead = new BNSHead();
        reqHead.id = requestMsgType;
        reqHead.version = 0;
        reqHead.log_id = new Random().nextInt();
        reqHead.body_len = requestBodyBytes.length;
        byte[] reqHeadByte = reqHead.toBytes();

        ByteBuffer byteBuffer = ByteBuffer.allocate(requestBodyBytes.length + reqHeadByte.length);
        byteBuffer.put(reqHeadByte);
        byteBuffer.put(requestBodyBytes);

        byte[] ret = byteBuffer.array();
        byteBuffer.clear();
        return ret;
    }

    protected byte[] readBNSMsg(InputStream is, short expectResponseType) throws IOException {
        byte[] respHeadByte = new byte[BNSHead.HEAD_LEN];
        int readCount = is.read(respHeadByte, 0, BNSHead.HEAD_LEN);
        if (readCount != BNSHead.HEAD_LEN) {
            throw new BNSException(ErrorCode.BNS_TALK_TO_AGENT_ERROR,
                    "Bad BNSHead, not enough head len, got" + readCount);
        }
        BNSHead respHead = BNSHead.fromBytes(respHeadByte);
        if (respHead == null || respHead.magic_num != BNSHead.MAGIC_NUM || respHead.body_len < 0
                || respHead.body_len > BNSHead.MAX_BODY_LEN) {
            throw new BNSException(ErrorCode.BNS_TALK_TO_AGENT_ERROR, "Bad BNSHead");
        }

        if (respHead.id != expectResponseType) {
            throw new BNSException(ErrorCode.BNS_TALK_TO_AGENT_ERROR,
                    "Not expected msg type, expect " + expectResponseType + ", but got" + respHead.id);
        }

        byte[] respBodyByte = new byte[respHead.body_len];
        readCount = 0;
        while (readCount < respHead.body_len) {
            int ret = is.read(respBodyByte, readCount, respHead.body_len - readCount);
            if (ret < 0) {
                throw new BNSException(ErrorCode.BNS_TALK_TO_AGENT_ERROR,
                        "Shorter body than expected, expect " + respHead.body_len + "B, but got " + readCount + "B");
            }
            readCount += ret;
        }

        return respBodyByte;
    }

    // Get the proper naming agent address
    // For those in VirtualMachine without a naming agent, go to ask its
    // gateway.
    public static String getLocalNamedAddress() {
        String localNamedAddress = LOCAL_NAMED_ADDR;
        String defaultGateway = getDefaultGateway();
        String jpaasv2Host = null;
        try {
            jpaasv2Host = System.getenv(JPAAS_HOST_NAME);
        } catch (Exception e) {
            /* do nothing here, just set jpaasv2Host null */
        }
        if (jpaasv2Host != null && jpaasv2Host.length() > 0) {
            localNamedAddress = jpaasv2Host;
        } else if ((defaultGateway != null) && defaultGateway.startsWith(LOCAL_IP_STARTS)) {
            localNamedAddress = defaultGateway;
        }
        return localNamedAddress;
    }

    public static String getDefaultGateway() {
        String defaultGateway = null;
        FileReader file = null;
        BufferedReader reader = null;
        try {
            file = new FileReader(PROC_NET_ROUTE_COMMAND);
            reader = new BufferedReader(file);
            // find the default gateway
            String line = null;
            while ((line = reader.readLine()) != null) {
                String[] tokens = line.split("\\s+");
                if (tokens.length <= 3) {
                    continue;
                }
                if (tokens[1].equals(DEFAULT_TOKEN)) {
                    String gatewayHex = tokens[2]; // 0102A8C0
                    if (gatewayHex.length() == 8) {
                        defaultGateway = SocketUtils.hexStringIp2DottedIp(gatewayHex);
                        return defaultGateway;
                    } else {
                        return null;
                    }
                }
            }

        } catch (Exception e) {
            LOGGER.debug(e.getMessage(), e.getCause());
            return null;
        } finally {
            if (file != null) {
                try {
                    file.close();
                } catch (IOException e) {
                    LOGGER.debug(e.getMessage(), e.getCause());
                }
            }
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    LOGGER.debug(e.getMessage(), e.getCause());
                }
            }
        }
        return defaultGateway;
    }

	@Override
	public List<Instance> queryInstances(String serviceName) throws BNSException {
		return getInstanceByService(serviceName);
	}

    /**
     * Gets the bns query proxy.
     *
     * @return the bns query proxy
     */
    public BNSQueryProxy getBnsQueryProxy() {
        return bnsQueryProxy;
    }

	
	
}
