/*
 * Copyright 2015 the original author or authors.
 *
 * Licensed under the BAIDU
 */
package com.baidu.driver4j.bns.spring;

import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import com.baidu.driver4j.bns.BNSCallback;
import com.baidu.driver4j.bns.BNSCallbackProxy;

/**
 * A auto register support bean for {@link BNSCallbackProxy}.
 * default autoRegister is true, it will  auto search all {@link BNSCallback} type class in the container and <br>
 * do register to the {@link BNSCallbackProxy}
 * 
 * @see BNSCallback
 * @see BNSCallbackProxy
 * @author xiemalin
 * @since 1.0.0
 */
public class BNSCallbackProxyBean extends BNSCallbackProxy
        implements ApplicationContextAware, InitializingBean, DisposableBean {
    
    /**
     * logger this class
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(BNSCallbackProxyBean.class.getName());

    private ApplicationContext applicationContext;
    
    /**
     * autoRegister if on will auto search all {@link BNSCallback} type class in the container and register in
     */
    private boolean autoRegister = true;

    /*
     * (non-Javadoc)
     * 
     * @see org.springframework.beans.factory.DisposableBean#destroy()
     */
    @Override
    public void destroy() throws Exception {
        stop();

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        start();
        
        if (autoRegister) {
            LOGGER.info("Starting auto register action for BNSCallback class type.");
            
            Map<String, BNSCallback> bnsCallbacks = applicationContext.getBeansOfType(BNSCallback.class);
            if (bnsCallbacks != null) {
                Iterator<Entry<String, BNSCallback>> iter = bnsCallbacks.entrySet().iterator();
                while (iter.hasNext()) {
                    Entry<String, BNSCallback> next = iter.next();
                    LOGGER.info("Auto register callback bean of bean name '" + next.getKey() + "'");
                    register(next.getValue());
                }
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.springframework.context.ApplicationContextAware#setApplicationContext(org.springframework.context.
     * ApplicationContext)
     */
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
    
    /**
     * setter method for property autoRegister
     * @param autoRegister the autoRegister to set
     */
    public void setAutoRegister(boolean autoRegister) {
        this.autoRegister = autoRegister;
    }
    
    /* (non-Javadoc)
     * @see com.baidu.driver4j.bns.BNSCallbackProxy#getCallbackMapping()
     */
    @Override
    protected Map<String, BNSCallback> getCallbackMapping() {
        return super.getCallbackMapping();
    }

}
