/**
 * 
 */
package com.baidu.driver4j.bns.spring;

import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.util.CollectionUtils;

import com.baidu.driver4j.bns.BNSQueryAgentProxy;
import com.baidu.driver4j.bns.Instance;

/**
 * Extends spring {@link PropertyPlaceholderConfigurer} to support get instance info from BNS service. <br>
 * example
 * {@code
 *  <bean class="com.baidu.driver4j.bns.spring.BNSPropertyPlaceholderConfigurer">
 *       <property name="bnsNamingSerivce">
 *           <set>
 *               <value>xml-noah.tc.all</value>
 *               <value>xml2-noah.tc.all</value>
 *           </set>
 *       </property>
 *       <property name="hostAndPortJoiner" value=":"></property>
 *       <property name="multiSplit" value=";"></property>
 *   </bean>
 * 
 * }
 * 
 * 
 * @author xiemalin
 * @since 1.0.0
 */
public class BNSPropertyPlaceholderConfigurer extends PropertyPlaceholderConfigurer implements InitializingBean {
    
    /**
     * logger this class
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(BNSCallbackProxyBean.class.getName());

    private Set<String> bnsNamingSerivce;
    
    private String multiSplit = ";";
    private String hostAndPortJoiner = ":";
    
    private BNSQueryAgentProxy bnsProxy = BNSQueryAgentProxy.proxy(); 
    
    
    /* (non-Javadoc)
     * @see org.springframework.beans.factory.config.PropertyPlaceholderConfigurer#processProperties(org.springframework.beans.factory.config.ConfigurableListableBeanFactory, java.util.Properties)
     */
    @Override
    protected void processProperties(ConfigurableListableBeanFactory beanFactoryToProcess, Properties props)
            throws BeansException {
        
        for (String serviceName : bnsNamingSerivce) {
            List<Instance> instances = bnsProxy.getInstanceByService(serviceName);
            if (CollectionUtils.isEmpty(instances)) {
                continue;
            }
            String instanceInfo = "";
            Iterator<Instance> iter = instances.iterator();
            while (iter.hasNext()) {
                Instance instance = iter.next();
                instanceInfo = instanceInfo + instance.getHostName() + hostAndPortJoiner + instance.getPort();
                
                if (iter.hasNext()) {
                    instanceInfo = instanceInfo + multiSplit;
                }
            }
            // put back to properties 
            props.put(serviceName, instanceInfo);
            LOGGER.info("Add service name '" + serviceName + "' from bns to placeholder.");

        }
        
        super.processProperties(beanFactoryToProcess, props);
    }
    
    public void setBnsNamingSerivce(Set<String> bnsNamingSerivce) {
        this.bnsNamingSerivce = bnsNamingSerivce;
    }
    
    public void setMultiSplit(String multiSplit) {
        this.multiSplit = multiSplit;
    }
    
    public void setHostAndPortJoiner(String hostAndPortJoiner) {
        this.hostAndPortJoiner = hostAndPortJoiner;
    }

    /* (non-Javadoc)
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        if (CollectionUtils.isEmpty(bnsNamingSerivce)) {
            throw new IllegalArgumentException("Property 'bnsNamingSerivce' is empty");
        }
        
    }
}
