/*
 * Copyright 2002-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.baidu.bjf.remoting.protobuf.utils;

import java.io.File;
import java.lang.reflect.Array;
import java.net.URL;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * A utility class for class and class loader.
 * 
 * @author xiemalin
 * @since 1.0.0
 */
public class ClassHelper {

    /**
     * package separator char
     */
    public static final char PACKAGE_SEPARATOR_CHAR = '.';

    /**
     * package separator string
     */
    public static final String PACKAGE_SEPARATOR = PACKAGE_SEPARATOR_CHAR + "";

    /**
     * get class internal name from Class.getName(). sub class like A$B to A.B
     * 
     * @param clsName class name
     * @return internalname
     */
    public static String getInternalName(String clsName) {
        if (clsName == null) {
            return null;
        }
        return clsName;
    }

    public static long getLastModifyTime(Class<?> cls) {
        if (cls == null) {
            return -1L;
        }

        String clsName = cls.getName().replace('.', '/') + ".class";
        
        ClassLoader classLoader = cls.getClassLoader();
        if (classLoader == null) {
        	return -1L;
        }
        
        URL resource = classLoader.getResource(clsName);
        if (resource != null) {
            return new File(resource.getFile()).lastModified();
        }

        return -1L;
    }

    /**
     * To fix name with thread context of class loader
     * 
     * @param name class name
     * @return loaded class
     * @throws ClassNotFoundException if class not found
     */
    public static Class<?> forNameWithThreadContextClassLoader(String name) throws ClassNotFoundException {
        return forName(name, Thread.currentThread().getContextClassLoader());
    }

    /**
     * to initialize clss by name from caller.
     * 
     * @param name class name
     * @param caller caller of Class type
     * @return loaded class
     * @throws ClassNotFoundException if class not found
     */
    public static Class<?> forNameWithCallerClassLoader(String name, Class<?> caller) throws ClassNotFoundException {
        return forName(name, caller.getClassLoader());
    }

    /**
     * get class loader from caller calss
     * 
     * @param caller caller of Class type
     * @return loaded class loader
     */
    public static ClassLoader getCallerClassLoader(Class<?> caller) {
        return caller.getClassLoader();
    }

    /**
     * get class loader
     * 
     * @param cls target class to get ownered class loader
     * @return class loader
     */
    public static ClassLoader getClassLoader(Class<?> cls) {
        ClassLoader cl = null;
        try {
            cl = Thread.currentThread().getContextClassLoader();
        } catch (Throwable ex) {
            // Cannot access thread context ClassLoader - falling back to system
            // class loader...
        }
        if (cl == null) {
            // No thread context class loader -> use class loader of this class.
            cl = cls.getClassLoader();
        }
        return cl;
    }

    /**
     * Return the default ClassLoader to use: typically the thread context ClassLoader, if available; the ClassLoader
     * that loaded the ClassUtils class will be used as fallback.
     * <p>
     * Call this method if you intend to use the thread context ClassLoader in a scenario where you absolutely need a
     * non-null ClassLoader reference: for example, for class path resource loading (but not necessarily for
     * <code>Class.forName</code>, which accepts a <code>null</code> ClassLoader reference as well).
     * 
     * @return the default ClassLoader (never <code>null</code>)
     * @see java.lang.Thread#getContextClassLoader()
     */
    public static ClassLoader getClassLoader() {
        return getClassLoader(ClassHelper.class);
    }

    /**
     * Same as <code>Class.forName()</code>, except that it works for primitive types.
     */
    public static Class<?> forName(String name) throws ClassNotFoundException {
        return forName(name, getClassLoader());
    }

    /**
     * Replacement for <code>Class.forName()</code> that also returns Class instances for primitives (like "int") and
     * array class names (like "String[]").
     * 
     * @param name the name of the Class
     * @param classLoader the class loader to use (may be <code>null</code>, which indicates the default class loader)
     * @return Class instance for the supplied name
     * @throws ClassNotFoundException if the class was not found
     * @throws LinkageError if the class file could not be loaded
     * @see Class#forName(String, boolean, ClassLoader)
     */
    public static Class<?> forName(String name, ClassLoader classLoader) throws ClassNotFoundException, LinkageError {

        Class<?> clazz = resolvePrimitiveClassName(name);
        if (clazz != null) {
            return clazz;
        }

        // "java.lang.String[]" style arrays
        if (name.endsWith(ARRAY_SUFFIX)) {
            String elementClassName = name.substring(0, name.length() - ARRAY_SUFFIX.length());
            Class<?> elementClass = forName(elementClassName, classLoader);
            return Array.newInstance(elementClass, 0).getClass();
        }

        // "[Ljava.lang.String;" style arrays
        int internalArrayMarker = name.indexOf(INTERNAL_ARRAY_PREFIX);
        if (internalArrayMarker != -1 && name.endsWith(";")) {
            String elementClassName = null;
            if (internalArrayMarker == 0) {
                elementClassName = name.substring(INTERNAL_ARRAY_PREFIX.length(), name.length() - 1);
            } else if (name.startsWith("[")) {
                elementClassName = name.substring(1);
            }
            Class<?> elementClass = forName(elementClassName, classLoader);
            return Array.newInstance(elementClass, 0).getClass();
        }

        ClassLoader classLoaderToUse = classLoader;
        if (classLoaderToUse == null) {
            classLoaderToUse = getClassLoader();
        }
        return classLoaderToUse.loadClass(name);
    }

    /**
     * Resolve the given class name as primitive class, if appropriate, according to the JVM's naming rules for
     * primitive classes.
     * <p>
     * Also supports the JVM's internal class names for primitive arrays. Does <i>not</i> support the "[]" suffix
     * notation for primitive arrays; this is only supported by {@link #forName}.
     * 
     * @param name the name of the potentially primitive class
     * @return the primitive class, or <code>null</code> if the name does not denote a primitive class or primitive
     *         array class
     */
    public static Class<?> resolvePrimitiveClassName(String name) {
        Class<?> result = null;
        // Most class names will be quite long, considering that they
        // SHOULD sit in a package, so a length check is worthwhile.
        if (name != null && name.length() <= 8) {
            // Could be a primitive - likely.
            result = (Class<?>) PRIMIIIVE_TYPE_NAME_MAP.get(name);
        }
        return result;
    }

    /** Suffix for array class names: "[]" */
    public static final String ARRAY_SUFFIX = "[]";
    /** Prefix for internal array class names: "[L" */
    private static final String INTERNAL_ARRAY_PREFIX = "[L";

    /**
     * Map with primitive type name as key and corresponding primitive type as value, for example: "int" -> "int.class".
     */
    private static final Map<String, Class<?>> PRIMIIIVE_TYPE_NAME_MAP = new HashMap<String, Class<?>>(16);

    /**
     * Map with primitive wrapper type as key and corresponding primitive type as value, for example: Integer.class ->
     * int.class.
     */
    private static final Map<Class<?>, Class<?>> PRIMITIVE_WRAPPER_TYPE_MAP = new HashMap<Class<?>, Class<?>>(8);

    static {
        PRIMITIVE_WRAPPER_TYPE_MAP.put(Boolean.class, boolean.class);
        PRIMITIVE_WRAPPER_TYPE_MAP.put(Byte.class, byte.class);
        PRIMITIVE_WRAPPER_TYPE_MAP.put(Character.class, char.class);
        PRIMITIVE_WRAPPER_TYPE_MAP.put(Double.class, double.class);
        PRIMITIVE_WRAPPER_TYPE_MAP.put(Float.class, float.class);
        PRIMITIVE_WRAPPER_TYPE_MAP.put(Integer.class, int.class);
        PRIMITIVE_WRAPPER_TYPE_MAP.put(Long.class, long.class);
        PRIMITIVE_WRAPPER_TYPE_MAP.put(Short.class, short.class);

        Set<Class<?>> primitiveTypeNames = new HashSet<Class<?>>(16);
        primitiveTypeNames.addAll(PRIMITIVE_WRAPPER_TYPE_MAP.values());
        primitiveTypeNames.addAll(Arrays.asList(new Class<?>[] { boolean[].class, byte[].class, char[].class,
                double[].class, float[].class, int[].class, long[].class, short[].class }));
        for (Iterator<Class<?>> it = primitiveTypeNames.iterator(); it.hasNext();) {
            Class<?> primitiveClass = (Class<?>) it.next();
            PRIMIIIVE_TYPE_NAME_MAP.put(primitiveClass.getName(), primitiveClass);
        }
    }

    /**
     * To get short message of object instance.
     * 
     * @param obj target object
     * @return short message
     */
    public static String toShortString(Object obj) {
        if (obj == null) {
            return "null";
        }
        return obj.getClass().getSimpleName() + "@" + System.identityHashCode(obj);

    }

    /**
     * get class simple name from {@link Class} instance. <br>
     * Here we should take case member class is a little different from common class.
     * 
     * @param cls target class
     * @return simple class name
     */
    public static String getClassName(Class<?> cls) {
        if (cls.isMemberClass()) {
            String name = cls.getName();
            name = StringUtils.substringAfterLast(name, PACKAGE_SEPARATOR);
            return name;
        }

        return cls.getSimpleName();
    }

    public static String getPackage(Class<?> cls) {
        Package pkg = cls.getPackage();
        // maybe null if package is blank or dynamic load class
        if (pkg == null) {
            String fullName = cls.getName();
            int index = fullName.lastIndexOf(PACKAGE_SEPARATOR_CHAR);
            if (index != -1) {
                return fullName.substring(0, index);
            }
            return "";
        }

        return pkg.getName();
    }

    /**
     * To test class if has default constructor method
     * 
     * @param cls target class
     * @return true if has default constructor method
     */
    public static boolean hasDefaultConstructor(Class<?> cls) {
        if (cls == null) {
            return false;
        }
        try {
            cls.getConstructor(new Class<?>[0]);
        } catch (NoSuchMethodException e2) {
            return false;
        } catch (SecurityException e2) {
            throw new IllegalArgumentException(e2.getMessage(), e2);
        }
        return true;
    }

    /**
     * get new class name with full package
     * 
     * @return class name
     */
    public static String getFullClassName(Class<?> cls) {
        return getPackage(cls) + PACKAGE_SEPARATOR + getClassName(cls);
    }
}
