/*
 * Copyright (c) 2014-2020 Baidu.com, Inc. All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.baidubce.http;

import java.util.Date;
import java.util.Set;

import org.apache.http.HttpHeaders;

import com.baidubce.BceClientConfiguration;
import com.baidubce.auth.BceV1Signer;
import com.baidubce.auth.SignOptions;
import com.baidubce.auth.Signer;
import com.baidubce.http.handler.ApiExplorerMetaResponseHandler;
import com.baidubce.http.handler.ApiExplorerResponseHandler;
import com.baidubce.http.handler.BceErrorResponseHandler;
import com.baidubce.http.handler.BceMetadataResponseHandler;
import com.baidubce.http.handler.HttpResponseHandler;
import com.baidubce.internal.InternalRequest;
import com.baidubce.model.ApiExplorerRequest;
import com.baidubce.model.ApiExplorerResponse;
import com.baidubce.util.DateUtils;
import com.google.common.collect.Sets;

/**
 * Provides the client for accessing the BCE Service.
 */
public class ApiExplorerClient {

    /**
     * Responsible for sending HTTP requests to services.
     */
    private BceHttpClient client;

    /**
     * A list of handlers for processing HTTP responses from services.
     *
     * @see com.baidubce.http.BceHttpClient#execute(InternalRequest, Class, HttpResponseHandler[])
     */
    private HttpResponseHandler[] responseHandlers;

    /**
     * The default http request content type for all BCE service APIs.
     */
    public static final String DEFAULT_CONTENT_TYPE = "application/json; charset=utf-8";

    /**
     * Constructs a new client to invoke BCE service methods with default configuration.
     */
    public ApiExplorerClient() {
        this(new BceV1Signer());
    }

    private static final Set<String> HEADERS_TO_SIGN = Sets.newHashSet("host", "x-bce-date");

    /**
     * Constructs a new client to invoke BCE service methods with singer.
     */
    public ApiExplorerClient(Signer signer) {
        client = new BceHttpClient(new BceClientConfiguration(), signer, false);
        responseHandlers = new HttpResponseHandler[] {
                new BceMetadataResponseHandler(),
                new ApiExplorerMetaResponseHandler(),
                new BceErrorResponseHandler(),
                new ApiExplorerResponseHandler()
        };
    }

    /**
     * Method for sending request to the target service.
     *
     * @param bceRequest The original request, as created by the user
     * @return The final response object.
     */
    public ApiExplorerResponse sendRequest(ApiExplorerRequest bceRequest) {
        InternalRequest request = bceRequest.getInternalRequest();
        if (!request.getHeaders().containsKey(HttpHeaders.CONTENT_TYPE)) {
            request.addHeader(HttpHeaders.CONTENT_TYPE, DEFAULT_CONTENT_TYPE);
        }
        if (!request.getHeaders().containsKey(HttpHeaders.DATE)) {
            request.addHeader(HttpHeaders.DATE, DateUtils.formatRfc822Date(new Date()));
        }
        SignOptions signOptions = new SignOptions();
        signOptions.setHeadersToSign(HEADERS_TO_SIGN);
        request.setSignOptions(signOptions);
        return client.execute(bceRequest.getInternalRequest(), ApiExplorerResponse.class, responseHandlers);
    }
}
