/*
 * Copyright (c) 2014-2019 Baidu.com, Inc. All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.baidubce.model;

import java.net.URI;
import java.nio.charset.StandardCharsets;

import com.baidubce.auth.BceCredentials;
import com.baidubce.auth.DefaultBceCredentials;
import com.baidubce.http.HttpMethodName;
import com.baidubce.internal.InternalRequest;
import com.baidubce.internal.RestartableInputStream;

/**
 * The original request, as created by the user
 */
public class ApiExplorerRequest {

    /**
     * A request object, populated with endpoint, resource path, ready for callers to populate
     * any additional headers or parameters, and execute.
     */
    private InternalRequest internalRequest;

    /**
     * Constructs a new ApiExplorerRequest
     *
     * @param httpMethodName The HTTP method to use when sending the request.
     * @param path           A complete request url without path variables
     */
    public ApiExplorerRequest(HttpMethodName httpMethodName, String path) {
        URI uri = URI.create(path);
        this.internalRequest = new InternalRequest(httpMethodName, uri);
    }

    public InternalRequest getInternalRequest() {
        return internalRequest;
    }

    public void setInternalRequest(InternalRequest internalRequest) {
        this.internalRequest = internalRequest;
    }

    /**
     * Create the BCE credentials and set into internalRequest
     *
     * @param accessKey The BCE secret access key for this credentials object.
     * @param secretKey The BCE secret access key for this credentials object.
     */
    public void setCredentials(String accessKey, String secretKey) {
        BceCredentials credentials = new DefaultBceCredentials(accessKey, secretKey);
        internalRequest.setCredentials(credentials);
    }

    public void addQueryParameter(String key, String value) {
        internalRequest.addParameter(key, value);
    }

    public void addHeaderParameter(String key, String value) {
        internalRequest.addHeader(key, value);
    }

    /**
     * The method to fill the internalRequest's content field with bceRequest.
     * Only support HttpMethodName.POST or HttpMethodName.PUT
     *
     * @param jsonBody The jsonBody content of request
     */
    public void setJsonBody(String jsonBody) {
        if (internalRequest.getHttpMethod() == HttpMethodName.POST
                || internalRequest.getHttpMethod() == HttpMethodName.PUT
                || internalRequest.getHttpMethod() == HttpMethodName.PATCH) {
            byte[] requestJson;
            requestJson = jsonBody.getBytes(StandardCharsets.UTF_8);
            internalRequest.setContent(RestartableInputStream.wrap(requestJson));
        }
    }

}
