/**
 * Copyright 2014-2023 the original author or authors
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.bernardomg.tabletop.dice.interpreter;

import java.util.function.Function;
import java.util.function.UnaryOperator;

import com.bernardomg.tabletop.dice.Dice;
import com.bernardomg.tabletop.dice.history.RollHistory;
import com.bernardomg.tabletop.dice.history.RollResult;
import com.bernardomg.tabletop.dice.notation.DiceNotationExpression;
import com.bernardomg.tabletop.dice.random.DiceToRollResult;
import com.bernardomg.tabletop.dice.random.NumberGenerator;
import com.bernardomg.tabletop.dice.random.RandomNumberGenerator;
import com.bernardomg.tabletop.dice.visitor.DiceRollAccumulator;

/**
 * Interpreter which simulates rolling the expression.
 * <p>
 * As some values, such as dice sets, represent random numbers the value returned may change each time this is applied
 * to an expression.
 * <p>
 * The random value will be generated by a {@link NumberGenerator}. By default a {@link RandomNumberGenerator} will be
 * used.
 * <p>
 * To customise the rolling procedure the constructors may receive some functions, one meant to simulate rolling the
 * dice, and the other to transform the results from rolling. This allows applying functions such as removing the lowest
 * result.
 *
 * @author Bernardo Mart&iacute;nez Garrido
 *
 */
public final class DiceRoller implements DiceInterpreter<RollHistory> {

    /**
     * Wrapped interpreter. Configured for the required operations.
     */
    private final DiceInterpreter<RollHistory> wrapped;

    /**
     * Default constructor.
     */
    public DiceRoller() {
        this(new DiceToRollResult());
    }

    /**
     * Constructs a transformer using the received roll generator for simulating rolls.
     *
     * @param roller
     *            the roller to use
     */
    public DiceRoller(final Function<Dice, RollResult> roller) {
        super();

        wrapped = new ConfigurableInterpreter<>(new PostorderTraverser(), new DiceRollAccumulator(roller));
    }

    /**
     * Constructs a transformer using the received roll generator for simulating rolls, chaining it to the received
     * transformer.
     *
     * @param roller
     *            the roller to use
     * @param transformer
     *            transformer to apply
     */
    public DiceRoller(final Function<Dice, RollResult> roller, final UnaryOperator<RollResult> transformer) {
        super();

        final Function<Dice, RollResult> finalRoller;

        finalRoller = roller.andThen(transformer);

        wrapped = new ConfigurableInterpreter<>(new PostorderTraverser(), new DiceRollAccumulator(finalRoller));
    }

    /**
     * Constructs a transformer using the received roller for simulating rolls.
     *
     * @param generator
     *            the random number generator to use
     */
    public DiceRoller(final NumberGenerator generator) {
        this(new DiceToRollResult(generator));
    }

    /**
     * Constructs a transformer using the received roller for simulating rolls and the received transformer on the
     * rolls.
     *
     * @param generator
     *            the random number generator to use
     * @param transformer
     *            transformer to apply
     */
    public DiceRoller(final NumberGenerator generator, final UnaryOperator<RollResult> transformer) {
        super();

        final Function<Dice, RollResult> finalRoller;

        finalRoller = new DiceToRollResult(generator).andThen(transformer);

        wrapped = new ConfigurableInterpreter<>(new PostorderTraverser(), new DiceRollAccumulator(finalRoller));
    }

    @Override
    public final RollHistory transform(final DiceNotationExpression expression) {
        return wrapped.transform(expression);
    }

}
