/*
 * Copyright 2014 - 2023 Blazebit.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.blazebit.persistence.view.impl.collection;

import com.blazebit.persistence.view.impl.entity.MapViewToEntityMapper;
import com.blazebit.persistence.view.impl.update.UpdateContext;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 *
 * @author Christian Beikov
 * @since 1.2.0
 */
public class MapRemoveAllKeysAction<C extends Map<K, V>, K, V> implements MapAction<C> {

    private final Collection<?> elements;
    private final Map<K, V> removedObjectsInView;

    public MapRemoveAllKeysAction(Collection<?> elements, Map<K, V> delegate) {
        this.elements = elements;
        this.removedObjectsInView = new LinkedHashMap<>(elements.size());
        for (Object key : elements) {
            V oldValue = delegate.get(key);
            if (oldValue != null) {
                this.removedObjectsInView.put((K) key, oldValue);
            }
        }
    }

    private MapRemoveAllKeysAction(List<?> elements, Map<K, V> removedObjectsInView) {
        this.elements = elements;
        this.removedObjectsInView = removedObjectsInView;
    }

    public static <C extends Map<K, V>, K, V> MapRemoveAllKeysAction<C, K, V> retainAll(Collection<?> c, C delegate) {
        int size = c.size() >= delegate.size() ? delegate.size() : delegate.size() - c.size();
        Map<K, V> removedObjectsInView = new LinkedHashMap<>(size);
        for (Map.Entry<K, V> entry : delegate.entrySet()) {
            if (!c.contains(entry.getKey())) {
                removedObjectsInView.put(entry.getKey(), entry.getValue());
            }
        }
        return new MapRemoveAllKeysAction<>(new ArrayList<>(removedObjectsInView.keySet()), removedObjectsInView);
    }

    @Override
    public void doAction(C map, UpdateContext context, MapViewToEntityMapper mapper, CollectionRemoveListener keyRemoveListener, CollectionRemoveListener valueRemoveListener) {
        if (mapper != null && mapper.getKeyMapper() != null) {
            if (elements.size() == 1) {
                Object e = elements.iterator().next();
                K key = (K) mapper.getKeyMapper().applyToEntity(context, null, e);
                V value = map.remove(key);
                if (value != null) {
                    if (keyRemoveListener != null) {
                        keyRemoveListener.onCollectionRemove(context, e);
                    }
                    if (valueRemoveListener != null) {
                        valueRemoveListener.onCollectionRemove(context, removedObjectsInView.get(e));
                    }
                }
            } else {
                List<Object> entities = new ArrayList<>(elements.size());
                entities.addAll(elements);
                mapper.getKeyMapper().applyAll(context, entities);
                int i = 0;
                for (Object e : elements) {
                    K key = (K) entities.get(i++);
                    V value = map.remove(key);
                    if (value != null) {
                        if (keyRemoveListener != null) {
                            keyRemoveListener.onCollectionRemove(context, e);
                        }
                        if (valueRemoveListener != null) {
                            valueRemoveListener.onCollectionRemove(context, removedObjectsInView.get(e));
                        }
                    }
                }
            }
        } else {
            if (map.size() > 0 && (keyRemoveListener != null || valueRemoveListener != null)) {
                for (Object k : elements) {
                    V v = map.remove(k);
                    if (v != null) {
                        if (keyRemoveListener != null) {
                            keyRemoveListener.onCollectionRemove(context, k);
                        }
                        if (valueRemoveListener != null) {
                            valueRemoveListener.onCollectionRemove(context, removedObjectsInView.get(k));
                        }
                    }
                }
            } else {
                map.keySet().removeAll(elements);
            }
        }
    }

    @Override
    public void undo(C map, Collection<?> removedKeys, Collection<?> addedKeys, Collection<?> removedElements, Collection<?> addedElements) {
        for (Map.Entry<K, V> entry : removedObjectsInView.entrySet()) {
            if (removedKeys.contains(entry.getKey()) || removedElements.contains(entry.getValue())) {
                map.put(entry.getKey(), entry.getValue());
            }
        }
    }

    @Override
    public Collection<Object> getAddedKeys() {
        return Collections.emptyList();
    }

    @Override
    public Collection<Object> getRemovedKeys() {
        return (Collection<Object>) removedObjectsInView.keySet();
    }

    @Override
    public Collection<Object> getAddedElements() {
        return Collections.emptyList();
    }

    @Override
    public Collection<Object> getRemovedElements() {
        return (Collection<Object>) removedObjectsInView.values();
    }

    @Override
    public Collection<Object> getAddedKeys(C collection) {
        return Collections.emptyList();
    }

    @Override
    public Collection<Object> getRemovedKeys(C collection) {
        List<Object> removedKeys = new ArrayList<>(elements.size());
        for (Object o : elements) {
            if (collection.containsKey(o)) {
                removedKeys.add(o);
            }
        }
        return removedKeys;
    }

    @Override
    public Collection<Object> getAddedElements(C collection) {
        return Collections.emptyList();
    }

    @Override
    public Collection<Object> getRemovedElements(C collection) {
        List<Object> removedElements = new ArrayList<>(elements.size());
        for (Object o : elements) {
            V value = collection.get(o);
            if (value != null) {
                removedElements.add(value);
            }
        }
        return removedElements;
    }

    @Override
    @SuppressWarnings("unchecked")
    public MapAction<C> replaceObject(Object oldKey, Object oldValue, Object newKey, Object newValue) {
        List<Object> newElements = RecordingUtils.replaceElements(elements, oldKey, newKey);

        if (newElements == null) {
            return null;
        }
        return new MapRemoveAllKeysAction(newElements, removedObjectsInView);
    }

    @Override
    @SuppressWarnings("unchecked")
    public MapAction<C> replaceObjects(Map<Object, Object> objectMapping) {
        if (objectMapping == null) {
            return this;
        }
        List<Object> newElements = RecordingUtils.replaceElements(elements, objectMapping);
        Map<Object, Object> newRemovedObjectsInView = RecordingUtils.replaceElements(removedObjectsInView, objectMapping);

        if (newElements != null) {
            if (newRemovedObjectsInView == null) {
                return new MapRemoveAllKeysAction(newElements, removedObjectsInView);
            } else {
                return new MapRemoveAllKeysAction(newElements, newRemovedObjectsInView);
            }
        } else if (newRemovedObjectsInView != null) {
            return new MapRemoveAllKeysAction(elements, newRemovedObjectsInView);
        } else {
            return this;
        }
    }

    @Override
    public void addAction(List<MapAction<C>> actions, Collection<Object> addedKeys, Collection<Object> removedKeys, Collection<Object> addedElements, Collection<Object> removedElements) {
        actions.add(this);
    }

}
