package com.blazebit.persistence.impl.hibernate;

import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;

import org.hibernate.AssertionFailure;
import org.hibernate.ScrollMode;
import org.hibernate.cfg.Settings;
import org.hibernate.engine.jdbc.spi.JdbcCoordinator;
import org.hibernate.engine.jdbc.spi.LogicalConnectionImplementor;
import org.hibernate.engine.jdbc.spi.SqlExceptionHelper;
import org.hibernate.engine.jdbc.spi.StatementPreparer;
import org.hibernate.engine.spi.SessionFactoryImplementor;

public class StatementPreparerImpl implements StatementPreparer {

    private JdbcCoordinator jdbcCoordinator;
    private SessionFactoryImplementor sessionFactoryImplementor;
    private boolean generated;
    private String[][] columns;
    private HibernateReturningResult<?> returningResult;

    public StatementPreparerImpl(JdbcCoordinator jdbcCoordinator, SessionFactoryImplementor sessionFactoryImplementor, boolean generated, String[][] columns, HibernateReturningResult<?> returningResult) {
        this.jdbcCoordinator = jdbcCoordinator;
        this.sessionFactoryImplementor = sessionFactoryImplementor;
        this.generated = generated;
        this.columns = columns;
        this.returningResult = returningResult;
    }

    protected final Settings settings() {
        return sessionFactoryImplementor.getSettings();
    }

    protected final Connection connection() {
        return logicalConnection().getConnection();
    }

    protected final LogicalConnectionImplementor logicalConnection() {
        return jdbcCoordinator.getLogicalConnection();
    }

    protected final SqlExceptionHelper sqlExceptionHelper() {
        return jdbcCoordinator.getTransactionCoordinator()
            .getTransactionContext()
            .getTransactionEnvironment()
            .getJdbcServices()
            .getSqlExceptionHelper();
    }

    @Override
    public Statement createStatement() {
        throw new UnsupportedOperationException("Not yet implemented!");
    }

    @Override
    public PreparedStatement prepareStatement(String sql) {
        throw new UnsupportedOperationException("Not yet implemented!");
    }

    @Override
    public PreparedStatement prepareStatement(String sql, final boolean isCallable) {
        throw new UnsupportedOperationException("Not yet implemented!");
    }

    private void checkAutoGeneratedKeysSupportEnabled() {
        if (!settings().isGetGeneratedKeysEnabled()) {
            throw new AssertionFailure("getGeneratedKeys() support is not enabled");
        }
    }

    @Override
    public PreparedStatement prepareStatement(String sql, final int autoGeneratedKeys) {
        throw new UnsupportedOperationException("Not yet implemented!");
    }

    @Override
    public PreparedStatement prepareStatement(String sql, final String[] columnNames) {
        throw new UnsupportedOperationException("Not yet implemented!");
    }

    @Override
    public PreparedStatement prepareQueryStatement(String sql, final boolean isCallable, final ScrollMode scrollMode) {
        checkAutoGeneratedKeysSupportEnabled();
        jdbcCoordinator.executeBatch();
        PreparedStatement ps = new QueryStatementPreparationTemplate(sql) {

            public PreparedStatement doPrepare() throws SQLException {
                if (generated) {
                    return connection().prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
                } else {
                    String[] columnNames = new String[columns.length];
                    for (int i = 0; i < columns.length; i++) {
                        columnNames[i] = columns[i][0];
                    }
                    return connection().prepareStatement(sql, columnNames);
                }
            }
        }.prepareStatement();
        ps = (PreparedStatement) Proxy.newProxyInstance(ps.getClass().getClassLoader(), new Class[]{ PreparedStatement.class }, new PreparedStatementInvocationHandler(ps, columns, returningResult));
        jdbcCoordinator.registerLastQuery(ps);
        return ps;
    }

    private abstract class StatementPreparationTemplate {

        protected final String sql;

        protected StatementPreparationTemplate(String sql) {
            this.sql = jdbcCoordinator.getTransactionCoordinator().getTransactionContext().onPrepareStatement(sql);
        }

        public PreparedStatement prepareStatement() {
            try {
                jdbcCoordinator.getLogicalConnection().getJdbcServices().getSqlStatementLogger().logStatement(sql);

                final PreparedStatement preparedStatement;
                try {
                    jdbcCoordinator.getTransactionCoordinator().getTransactionContext().startPrepareStatement();
                    preparedStatement = doPrepare();
                    setStatementTimeout(preparedStatement);
                } finally {
                    jdbcCoordinator.getTransactionCoordinator().getTransactionContext().endPrepareStatement();
                }
                postProcess(preparedStatement);
                return preparedStatement;
            } catch (SQLException e) {
                throw sqlExceptionHelper().convert(e, "could not prepare statement", sql);
            }
        }

        protected abstract PreparedStatement doPrepare() throws SQLException;

        public void postProcess(PreparedStatement preparedStatement) throws SQLException {
            jdbcCoordinator.register(preparedStatement);
            logicalConnection().notifyObserversStatementPrepared();
        }

        private void setStatementTimeout(PreparedStatement preparedStatement) throws SQLException {
            final int remainingTransactionTimeOutPeriod = jdbcCoordinator.determineRemainingTransactionTimeOutPeriod();
            if (remainingTransactionTimeOutPeriod > 0) {
                preparedStatement.setQueryTimeout(remainingTransactionTimeOutPeriod);
            }
        }
    }

    private abstract class QueryStatementPreparationTemplate extends StatementPreparationTemplate {

        protected QueryStatementPreparationTemplate(String sql) {
            super(sql);
        }

        public void postProcess(PreparedStatement preparedStatement) throws SQLException {
            super.postProcess(preparedStatement);
            setStatementFetchSize(preparedStatement);
        }
    }

    private void setStatementFetchSize(PreparedStatement statement) throws SQLException {
        if (settings().getJdbcFetchSize() != null) {
            statement.setFetchSize(settings().getJdbcFetchSize());
        }
    }

}
