/*

Copyright (C) SYSTAP, LLC DBA Blazegraph 2006-2016.  All rights reserved.

Contact:
     SYSTAP, LLC DBA Blazegraph
     2501 Calvert ST NW #106
     Washington, DC 20008
     licenses@blazegraph.com

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
/*
 * Created on Jun 23, 2008
 */

package com.bigdata.relation.rule;

import java.util.Iterator;

import com.bigdata.bop.IPredicate;
import com.bigdata.relation.IMutableRelation;
import com.bigdata.relation.rule.eval.ISolution;

/**
 * A program consists of a set of rules and/or programs. Some programs are
 * executed sequentially while others are (at least logically) parallel. A
 * program may also specify the transitive closure of its rules.
 * 
 * @todo add an XML (de-)serialization for programs and rules so that the rule
 *       sets may be declared. Note that there are a few very specialized rules
 *       whose semantics might not be fully declarable for the fast closure
 *       method used by the RDF KB.
 * 
 * @todo Make it possible for people to easily extend the RDF KB rule sets or
 *       run rules over a RDF data set, but note that there are interactions in
 *       the rules choosen for evaluation during forward closure and those
 *       choosen for evaluation at query time.
 * 
 * @author <a href="mailto:thompsonbry@users.sourceforge.net">Bryan Thompson</a>
 * @version $Id$
 */
public interface IProgram extends IStep {

    /**
     * <code>true</code> iff the {@link #steps()} MAY be executed in parallel.
     * When <code>false</code> the steps WILL be executed sequentially and
     * (for mutation) the buffered writes will be flushed after each step.
     * <p>
     * Note: Sequential execution of a program makes a multi-{@link IRule}
     * program MUCH easier to debug as the rules are executed one by one and the
     * buffer is flushed after each rule so you can directly see the #of
     * elements consider by the {@link IRule} for each {@link IPredicate} in the
     * tail and the #of {@link ISolution}s generated by the {@link IRule}.
     */
    boolean isParallel();

    /**
     * <code>true</code> iff the fixed point closure of the {@link IProgram}
     * should be computed.
     * <p>
     * Note: Closure is computed in rounds. The rounds continue until the
     * {@link IMutableRelation}(s) on which the steps are writing reach a fixed
     * point (no new solutions are added to the relation(s) within a given
     * round). If a program has a fixed point, then the same closure will be
     * obtained by either sequential or parallel execution of the steps within
     * each round and in general parallel execution of the steps is perferred as
     * it has better performance. However, the steps MAY be executed
     * sequentially to facilitate debugging.
     */
    boolean isClosure();
    
    /**
     * The sequence of sub-program {@link IStep}s.
     */
    Iterator<IStep> steps();

    /**
     * The #of steps in the program (non-recursive).
     */
    public int stepCount();

    /**
     * An array containing the steps in the program (non-recursive).
     */
    public IStep[] toArray();

}
