/*
 * Decompiled with CFR 0.152.
 */
package com.bloxbean.cardano.client.address;

import com.bloxbean.cardano.client.address.Address;
import com.bloxbean.cardano.client.address.AddressType;
import com.bloxbean.cardano.client.address.Credential;
import com.bloxbean.cardano.client.address.CredentialType;
import com.bloxbean.cardano.client.address.Pointer;
import com.bloxbean.cardano.client.address.util.AddressEncoderDecoderUtil;
import com.bloxbean.cardano.client.common.model.Network;
import com.bloxbean.cardano.client.crypto.Bech32;
import com.bloxbean.cardano.client.crypto.Blake2bUtil;
import com.bloxbean.cardano.client.crypto.bip32.key.HdPublicKey;
import com.bloxbean.cardano.client.crypto.bip32.util.BytesUtil;
import com.bloxbean.cardano.client.crypto.cip1852.CIP1852;
import com.bloxbean.cardano.client.crypto.cip1852.DerivationPath;
import com.bloxbean.cardano.client.exception.AddressRuntimeException;
import com.bloxbean.cardano.client.exception.CborSerializationException;
import com.bloxbean.cardano.client.spec.NetworkId;
import com.bloxbean.cardano.client.spec.Script;
import com.google.common.primitives.Bytes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AddressProvider {
    private static final Logger log = LoggerFactory.getLogger(AddressProvider.class);
    private static final byte BASE_PAYMENT_KEY_STAKE_KEY_HEADER_TYPE = 0;
    private static final byte BASE_PAYMENT_SCRIPT_STAKE_KEY_HEADER_TYPE = 16;
    private static final byte BASE_PAYMENT_KEY_STAKE_SCRIPT_HEADER_TYPE = 32;
    private static final byte BASE_PAYMENT_SCRIPT_STAKE_SCRIPT_HEADER_TYPE = 48;
    private static final byte PTR_PAYMENT_KEY_STAKE_PTR_HEADER_TYPE = 64;
    private static final byte PTR_PAYMENT_SCRIPT_STAKE_PTR_HEADER_TYPE = 80;
    private static final byte ENT_PAYMENT_KEY_HEADER_TYPE = 96;
    private static final byte ENT_PAYMENT_SCRIPT_HEADER_TYPE = 112;
    private static final byte RWD_STAKE_KEY_HEDER_TYPE = -32;
    private static final byte RWD_STAKE_SCRIPT_HEADER_TYPE = -16;

    public static Address getBaseAddress(HdPublicKey paymentKey, HdPublicKey delegationKey, Network networkInfo) {
        if (paymentKey == null || delegationKey == null) {
            throw new AddressRuntimeException("paymentkey and delegationKey cannot be null");
        }
        byte[] paymentKeyHash = paymentKey.getKeyHash();
        byte[] delegationKeyHash = delegationKey.getKeyHash();
        return AddressProvider.getAddress(paymentKeyHash, delegationKeyHash, (byte)0, networkInfo, AddressType.Base);
    }

    public static Address getBaseAddress(Script paymentScript, HdPublicKey delegationKey, Network networkInfo) {
        byte[] paymentScriptHash;
        if (paymentScript == null || delegationKey == null) {
            throw new AddressRuntimeException("paymentScript and delegationKey cannot be null");
        }
        try {
            paymentScriptHash = paymentScript.getScriptHash();
        }
        catch (CborSerializationException e) {
            throw new AddressRuntimeException("Unable to get script hash from payment script", (Exception)((Object)e));
        }
        byte[] delegationKeyHash = delegationKey.getKeyHash();
        byte headerType = 16;
        return AddressProvider.getAddress(paymentScriptHash, delegationKeyHash, headerType, networkInfo, AddressType.Base);
    }

    public static Address getBaseAddress(HdPublicKey paymentKey, Script delegationScript, Network networkInfo) {
        byte[] delegationScriptHash;
        if (paymentKey == null || delegationScript == null) {
            throw new AddressRuntimeException("paymentkey and delegationScript cannot be null");
        }
        byte[] paymentKeyHash = paymentKey.getKeyHash();
        try {
            delegationScriptHash = delegationScript.getScriptHash();
        }
        catch (CborSerializationException e) {
            throw new AddressRuntimeException("Unable to get script hash from delegation script", (Exception)((Object)e));
        }
        byte headerType = 32;
        return AddressProvider.getAddress(paymentKeyHash, delegationScriptHash, headerType, networkInfo, AddressType.Base);
    }

    public static Address getBaseAddress(Script paymentScript, Script delegationScript, Network networkInfo) {
        byte[] delegationScriptHash;
        byte[] paymentScriptHash;
        if (paymentScript == null || delegationScript == null) {
            throw new AddressRuntimeException("paymentScript and delegationScript cannot be null");
        }
        try {
            paymentScriptHash = paymentScript.getScriptHash();
            delegationScriptHash = delegationScript.getScriptHash();
        }
        catch (CborSerializationException e) {
            throw new RuntimeException(e);
        }
        byte headerType = 48;
        return AddressProvider.getAddress(paymentScriptHash, delegationScriptHash, headerType, networkInfo, AddressType.Base);
    }

    public static Address getBaseAddress(Credential paymentCredential, Credential delegationCredential, Network networkInfo) {
        if (paymentCredential == null || delegationCredential == null) {
            throw new AddressRuntimeException("paymentCredential and delegationCredential cannot be null");
        }
        if (paymentCredential.getType() == CredentialType.Key && delegationCredential.getType() == CredentialType.Key) {
            return AddressProvider.getAddress(paymentCredential.getBytes(), delegationCredential.getBytes(), (byte)0, networkInfo, AddressType.Base);
        }
        if (paymentCredential.getType() == CredentialType.Script && delegationCredential.getType() == CredentialType.Key) {
            return AddressProvider.getAddress(paymentCredential.getBytes(), delegationCredential.getBytes(), (byte)16, networkInfo, AddressType.Base);
        }
        if (paymentCredential.getType() == CredentialType.Key && delegationCredential.getType() == CredentialType.Script) {
            return AddressProvider.getAddress(paymentCredential.getBytes(), delegationCredential.getBytes(), (byte)32, networkInfo, AddressType.Base);
        }
        if (paymentCredential.getType() == CredentialType.Script && delegationCredential.getType() == CredentialType.Script) {
            return AddressProvider.getAddress(paymentCredential.getBytes(), delegationCredential.getBytes(), (byte)48, networkInfo, AddressType.Base);
        }
        throw new AddressRuntimeException("Invalid credential type, should be either Key or Script. Payment Credential: " + paymentCredential + ", Delegation Credential: " + delegationCredential);
    }

    public static Address getPointerAddress(HdPublicKey paymentKey, Pointer delegationPointer, Network networkInfo) {
        if (paymentKey == null || delegationPointer == null) {
            throw new AddressRuntimeException("paymentkey and delegationKey cannot be null");
        }
        byte[] paymentKeyHash = paymentKey.getKeyHash();
        byte[] delegationPointerHash = BytesUtil.merge((byte[][])new byte[][]{AddressProvider.variableNatEncode(delegationPointer.slot), AddressProvider.variableNatEncode(delegationPointer.txIndex), AddressProvider.variableNatEncode(delegationPointer.certIndex)});
        byte headerType = 64;
        return AddressProvider.getAddress(paymentKeyHash, delegationPointerHash, headerType, networkInfo, AddressType.Ptr);
    }

    public static Address getPointerAddress(Script paymentScript, Pointer delegationPointer, Network networkInfo) {
        byte[] paymentScriptHash;
        if (paymentScript == null || delegationPointer == null) {
            throw new AddressRuntimeException("paymentScript and delegationKey cannot be null");
        }
        try {
            paymentScriptHash = paymentScript.getScriptHash();
        }
        catch (CborSerializationException e) {
            throw new AddressRuntimeException("Unable to get script hash from payment script", (Exception)((Object)e));
        }
        byte[] delegationPointerHash = BytesUtil.merge((byte[][])new byte[][]{AddressProvider.variableNatEncode(delegationPointer.slot), AddressProvider.variableNatEncode(delegationPointer.txIndex), AddressProvider.variableNatEncode(delegationPointer.certIndex)});
        byte headerType = 80;
        return AddressProvider.getAddress(paymentScriptHash, delegationPointerHash, headerType, networkInfo, AddressType.Ptr);
    }

    public static Address getPointerAddress(Credential paymentCredential, Pointer delegationPointer, Network networkInfo) {
        if (paymentCredential == null || delegationPointer == null) {
            throw new AddressRuntimeException("paymentCredential and delegationPointer cannot be null");
        }
        byte[] delegationPointerHash = BytesUtil.merge((byte[][])new byte[][]{AddressProvider.variableNatEncode(delegationPointer.slot), AddressProvider.variableNatEncode(delegationPointer.txIndex), AddressProvider.variableNatEncode(delegationPointer.certIndex)});
        switch (paymentCredential.getType()) {
            case Key: {
                return AddressProvider.getAddress(paymentCredential.getBytes(), delegationPointerHash, (byte)64, networkInfo, AddressType.Ptr);
            }
            case Script: {
                return AddressProvider.getAddress(paymentCredential.getBytes(), delegationPointerHash, (byte)80, networkInfo, AddressType.Ptr);
            }
        }
        throw new AddressRuntimeException("Invalid credential type, should be either Key or Script. Payment Credential: " + paymentCredential + ", Delegation Pointer: " + delegationPointer);
    }

    public static Address getEntAddress(HdPublicKey paymentKey, Network networkInfo) {
        if (paymentKey == null) {
            throw new AddressRuntimeException("paymentkey cannot be null");
        }
        byte[] paymentKeyHash = paymentKey.getKeyHash();
        byte headerType = 96;
        return AddressProvider.getAddress(paymentKeyHash, null, headerType, networkInfo, AddressType.Enterprise);
    }

    public static Address getEntAddress(Script paymentScript, Network networkInfo) {
        byte[] paymentScriptHash;
        if (paymentScript == null) {
            throw new AddressRuntimeException("paymentScript cannot be null");
        }
        try {
            paymentScriptHash = paymentScript.getScriptHash();
        }
        catch (CborSerializationException e) {
            throw new AddressRuntimeException("Unable to get script hash from payment script", (Exception)((Object)e));
        }
        byte headerType = 112;
        return AddressProvider.getAddress(paymentScriptHash, null, headerType, networkInfo, AddressType.Enterprise);
    }

    public static Address getEntAddress(@NonNull Credential paymentCredential, Network networkInfo) {
        if (paymentCredential == null) {
            throw new NullPointerException("paymentCredential is marked non-null but is null");
        }
        switch (paymentCredential.getType()) {
            case Key: {
                return AddressProvider.getAddress(paymentCredential.getBytes(), null, (byte)96, networkInfo, AddressType.Enterprise);
            }
            case Script: {
                return AddressProvider.getAddress(paymentCredential.getBytes(), null, (byte)112, networkInfo, AddressType.Enterprise);
            }
        }
        throw new AddressRuntimeException("Invalid credential type, should be either Key or Script. Payment Credential: " + paymentCredential);
    }

    public static Address getRewardAddress(HdPublicKey delegationKey, Network networkInfo) {
        if (delegationKey == null) {
            throw new AddressRuntimeException("stakeKey cannot be null");
        }
        byte[] stakeKeyHash = delegationKey.getKeyHash();
        byte headerType = -32;
        return AddressProvider.getAddress(null, stakeKeyHash, headerType, networkInfo, AddressType.Reward);
    }

    public static Address getRewardAddress(Script delegationScript, Network networkInfo) {
        byte[] stakeScriptHash;
        if (delegationScript == null) {
            throw new AddressRuntimeException("delegationScript cannot be null");
        }
        try {
            stakeScriptHash = delegationScript.getScriptHash();
        }
        catch (CborSerializationException e) {
            throw new AddressRuntimeException("Unable to get script hash from delegation script", (Exception)((Object)e));
        }
        byte headerType = -16;
        return AddressProvider.getAddress(null, stakeScriptHash, headerType, networkInfo, AddressType.Reward);
    }

    public static Address getRewardAddress(@NonNull Credential stakeCredential, Network networkInfo) {
        if (stakeCredential == null) {
            throw new NullPointerException("stakeCredential is marked non-null but is null");
        }
        switch (stakeCredential.getType()) {
            case Key: {
                return AddressProvider.getAddress(null, stakeCredential.getBytes(), (byte)-32, networkInfo, AddressType.Reward);
            }
            case Script: {
                return AddressProvider.getAddress(null, stakeCredential.getBytes(), (byte)-16, networkInfo, AddressType.Reward);
            }
        }
        throw new AddressRuntimeException("Invalid credential type, should be either Key or Script. Stake Credential: " + stakeCredential);
    }

    private static Address getAddress(byte[] paymentKeyHash, byte[] stakeKeyHash, byte headerKind, Network networkInfo, AddressType addressType) {
        NetworkId network = AddressEncoderDecoderUtil.getNetworkId(networkInfo);
        String prefix = AddressEncoderDecoderUtil.getPrefixHeader(addressType) + AddressEncoderDecoderUtil.getPrefixTail(network);
        byte header = AddressEncoderDecoderUtil.getAddressHeader(headerKind, networkInfo, addressType);
        byte[] addressArray = AddressProvider.getAddressBytes(paymentKeyHash, stakeKeyHash, addressType, header);
        return new Address(prefix, addressArray);
    }

    private static byte[] getAddressBytes(byte[] paymentKeyHash, byte[] stakeKeyHash, AddressType addressType, byte header) {
        byte[] addressArray;
        switch (addressType) {
            case Base: {
                addressArray = new byte[1 + paymentKeyHash.length + stakeKeyHash.length];
                addressArray[0] = header;
                System.arraycopy(paymentKeyHash, 0, addressArray, 1, paymentKeyHash.length);
                System.arraycopy(stakeKeyHash, 0, addressArray, paymentKeyHash.length + 1, stakeKeyHash.length);
                break;
            }
            case Enterprise: {
                addressArray = new byte[1 + paymentKeyHash.length];
                addressArray[0] = header;
                System.arraycopy(paymentKeyHash, 0, addressArray, 1, paymentKeyHash.length);
                break;
            }
            case Reward: {
                addressArray = new byte[1 + stakeKeyHash.length];
                addressArray[0] = header;
                System.arraycopy(stakeKeyHash, 0, addressArray, 1, stakeKeyHash.length);
                break;
            }
            case Ptr: {
                addressArray = new byte[1 + paymentKeyHash.length + stakeKeyHash.length];
                addressArray[0] = header;
                System.arraycopy(paymentKeyHash, 0, addressArray, 1, paymentKeyHash.length);
                System.arraycopy(stakeKeyHash, 0, addressArray, paymentKeyHash.length + 1, stakeKeyHash.length);
                break;
            }
            default: {
                throw new AddressRuntimeException("Unknown address type");
            }
        }
        return addressArray;
    }

    public static boolean verifyAddress(@NonNull Address address, byte[] publicKey) {
        byte[] newAddressBytes;
        if (address == null) {
            throw new NullPointerException("address is marked non-null but is null");
        }
        String prefix = address.getPrefix();
        AddressType addressType = address.getAddressType();
        byte[] addressBytes = address.getBytes();
        byte header = addressBytes[0];
        if (AddressType.Reward.equals((Object)addressType)) {
            byte[] stakeKeyHash = Blake2bUtil.blake2bHash224((byte[])publicKey);
            newAddressBytes = AddressProvider.getAddressBytes(null, stakeKeyHash, addressType, header);
        } else {
            byte[] stakeKeyHash = AddressProvider.getDelegationCredentialHash(address).orElse(null);
            byte[] paymentKeyHash = Blake2bUtil.blake2bHash224((byte[])publicKey);
            newAddressBytes = AddressProvider.getAddressBytes(paymentKeyHash, stakeKeyHash, addressType, header);
        }
        Address newAddress = new Address(prefix, newAddressBytes);
        if (log.isDebugEnabled()) {
            log.debug("Address to compare : " + address.toBech32());
            log.debug("Address derived from pub key : " + newAddress.toBech32());
        }
        return newAddress.toBech32().equals(address.toBech32());
    }

    public static Address getStakeAddress(@NonNull Address address) {
        if (address == null) {
            throw new NullPointerException("address is marked non-null but is null");
        }
        if (AddressType.Base != address.getAddressType()) {
            throw new AddressRuntimeException(String.format("Stake address can't be derived. Required address type: Base Address, Found: %s ", new Object[]{address.getAddressType()}));
        }
        byte[] stakeKeyHash = AddressProvider.getDelegationCredentialHash(address).orElseThrow(() -> new AddressRuntimeException("StakeKeyHash was not found for the address"));
        AddressType addressType = AddressType.Reward;
        byte[] addressBytes = address.getBytes();
        byte header = addressBytes[0];
        int stakeHeader = (header & 0x20) > 0 ? 240 : 224;
        int network = header & 0xF;
        byte[] rewardAddressBytes = AddressProvider.getAddressBytes(null, stakeKeyHash, addressType, (byte)(stakeHeader |= network));
        return new Address(rewardAddressBytes);
    }

    public static Address getStakeAddressFromAccountPublicKey(String accountPubKey, Network network) {
        Objects.requireNonNull(accountPubKey, "accountPubKey cannot be null");
        Objects.requireNonNull(network, "network cannot be null");
        if (!accountPubKey.startsWith("acct_xvk") && !accountPubKey.startsWith("xpub")) {
            throw new IllegalArgumentException("Invalid account public key. Must start with 'acct_xvk' or 'xpub'");
        }
        byte[] accountPubKeyBytes = Bech32.decode((String)accountPubKey).data;
        return AddressProvider.getStakeAddressFromAccountPublicKey(accountPubKeyBytes, network);
    }

    public static Address getStakeAddressFromAccountPublicKey(byte[] accountPubKeyBytes, Network network) {
        Objects.requireNonNull(accountPubKeyBytes, "accountPubKeyBytes cannot be null");
        Objects.requireNonNull(network, "network cannot be null");
        if (accountPubKeyBytes.length != 64) {
            throw new IllegalArgumentException("Invalid account public key");
        }
        HdPublicKey stakeHdPubKey = new CIP1852().getPublicKeyFromAccountPubKey(accountPubKeyBytes, DerivationPath.createStakeAddressDerivationPath());
        return AddressProvider.getRewardAddress(stakeHdPubKey, network);
    }

    public static Optional<byte[]> getDelegationCredentialHash(Address address) {
        byte[] stakeKeyHash;
        AddressType addressType = address.getAddressType();
        byte[] addressBytes = address.getBytes();
        switch (addressType) {
            case Base: {
                stakeKeyHash = new byte[28];
                System.arraycopy(addressBytes, 29, stakeKeyHash, 0, stakeKeyHash.length);
                break;
            }
            case Enterprise: {
                stakeKeyHash = null;
                break;
            }
            case Reward: {
                stakeKeyHash = new byte[28];
                System.arraycopy(addressBytes, 1, stakeKeyHash, 0, stakeKeyHash.length);
                break;
            }
            case Ptr: {
                stakeKeyHash = new byte[addressBytes.length - 1 - 28];
                System.arraycopy(addressBytes, 29, stakeKeyHash, 0, stakeKeyHash.length);
                break;
            }
            default: {
                throw new AddressRuntimeException("DelegationHash can't be found for address type : " + addressType);
            }
        }
        return Optional.ofNullable(stakeKeyHash);
    }

    public static Optional<byte[]> getPaymentCredentialHash(Address address) {
        byte[] paymentKeyHash;
        AddressType addressType = address.getAddressType();
        byte[] addressBytes = address.getBytes();
        switch (addressType) {
            case Base: 
            case Enterprise: 
            case Ptr: {
                paymentKeyHash = new byte[28];
                System.arraycopy(addressBytes, 1, paymentKeyHash, 0, paymentKeyHash.length);
                break;
            }
            case Reward: {
                paymentKeyHash = null;
                break;
            }
            default: {
                throw new AddressRuntimeException("Unable to get payment key hash for address type: " + addressType + ", address=" + address.getAddress());
            }
        }
        return Optional.ofNullable(paymentKeyHash);
    }

    public static Optional<Credential> getDelegationCredential(Address address) {
        Objects.requireNonNull(address, "address cannot be null");
        if (AddressProvider.isStakeKeyHashInDelegationPart(address)) {
            return AddressProvider.getDelegationCredentialHash(address).map(hash -> Credential.fromKey(hash));
        }
        if (AddressProvider.isScriptHashInDelegationPart(address)) {
            return AddressProvider.getDelegationCredentialHash(address).map(hash -> Credential.fromScript(hash));
        }
        return Optional.empty();
    }

    public static Optional<Credential> getPaymentCredential(Address address) {
        Objects.requireNonNull(address, "address cannot be null");
        if (AddressProvider.isPubKeyHashInPaymentPart(address)) {
            return AddressProvider.getPaymentCredentialHash(address).map(hash -> Credential.fromKey(hash));
        }
        if (AddressProvider.isScriptHashInPaymentPart(address)) {
            return AddressProvider.getPaymentCredentialHash(address).map(hash -> Credential.fromScript(hash));
        }
        return Optional.empty();
    }

    public static boolean isPubKeyHashInPaymentPart(@NonNull Address address) {
        if (address == null) {
            throw new NullPointerException("address is marked non-null but is null");
        }
        AddressType addressType = address.getAddressType();
        if (addressType != AddressType.Base && addressType != AddressType.Enterprise && addressType != AddressType.Ptr) {
            if (log.isDebugEnabled()) {
                log.warn("Method not supported for address type=" + addressType + ", address=" + address.getAddress());
            }
            return false;
        }
        byte[] addressBytes = address.getBytes();
        byte header = addressBytes[0];
        return (header & 0x10) == 0;
    }

    public static boolean isScriptHashInPaymentPart(@NonNull Address address) {
        if (address == null) {
            throw new NullPointerException("address is marked non-null but is null");
        }
        AddressType addressType = address.getAddressType();
        if (addressType != AddressType.Base && addressType != AddressType.Enterprise && addressType != AddressType.Ptr) {
            if (log.isDebugEnabled()) {
                log.warn("Method not supported for address type=" + addressType + ", address=" + address.getAddress());
            }
            return false;
        }
        return !AddressProvider.isPubKeyHashInPaymentPart(address);
    }

    public static boolean isStakeKeyHashInDelegationPart(@NonNull Address address) {
        if (address == null) {
            throw new NullPointerException("address is marked non-null but is null");
        }
        AddressType addressType = address.getAddressType();
        if (addressType != AddressType.Base && addressType != AddressType.Ptr && addressType != AddressType.Reward) {
            if (log.isDebugEnabled()) {
                log.warn("Method not supported for address type=" + addressType + ", address=" + address.getAddress());
            }
            return false;
        }
        byte[] addressBytes = address.getBytes();
        byte header = addressBytes[0];
        if (addressType == AddressType.Reward) {
            return (header & 0x10) == 0;
        }
        return (header & 0x20) == 0;
    }

    public static boolean isScriptHashInDelegationPart(@NonNull Address address) {
        if (address == null) {
            throw new NullPointerException("address is marked non-null but is null");
        }
        AddressType addressType = address.getAddressType();
        if (addressType != AddressType.Base && addressType != AddressType.Ptr && addressType != AddressType.Reward) {
            if (log.isDebugEnabled()) {
                log.warn("Method not supported for address type=" + addressType + ", address=" + address.getAddress());
            }
            return false;
        }
        return !AddressProvider.isStakeKeyHashInDelegationPart(address);
    }

    private static byte[] variableNatEncode(long num) {
        ArrayList<Byte> output = new ArrayList<Byte>();
        output.add((byte)(num & 0x7FL));
        num /= 128L;
        while (num > 0L) {
            output.add((byte)(num & 0x7FL | 0x80L));
            num /= 128L;
        }
        Collections.reverse(output);
        return Bytes.toArray(output);
    }
}

