/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.bmuschko.gradle.docker.tasks.container

import org.gradle.api.tasks.Input
import org.gradle.api.tasks.Optional
import org.gradle.api.tasks.OutputFile

import java.util.zip.GZIPOutputStream

class DockerCopyFileFromContainer extends DockerExistingContainer {
    @Input
    String resource

    @Input
    @OutputFile
    File destFile

    @Input
    @Optional
    Boolean compressed = Boolean.FALSE

    @Override
    void runRemoteCommand(dockerClient) {
        logger.quiet "Copying '${getResource()}' from container with ID '${getContainerId()}' to '${getDestFile()}'."

        getDestFile().withOutputStream  { OutputStream out ->
            def input

            try {
                input = dockerClient.copyFileFromContainerCmd(getContainerId(), getResource()).exec()

                if (getCompressed()) {
                    compressFile(out, input)
                } else {
                    out << input
                }
            }
            finally {
                input?.close()
            }
        }
    }

    private void compressFile(OutputStream outputStream, input) {
        GZIPOutputStream gzipOut

        try {
            gzipOut = new GZIPOutputStream(outputStream)
            gzipOut << input
        }
        finally {
            gzipOut?.close()
        }
    }
}
