/*
 * Decompiled with CFR 0.152.
 */
package com.box.sdk;

import com.box.sdk.BackoffCounter;
import com.box.sdk.BoxAPIConnection;
import com.box.sdk.BoxAPIException;
import com.box.sdk.BoxAPIResponse;
import com.box.sdk.BoxFileUploadSession;
import com.box.sdk.BoxFileUploadSessionPart;
import com.box.sdk.BoxGlobalSettings;
import com.box.sdk.BoxJSONResponse;
import com.box.sdk.BoxLogger;
import com.box.sdk.BoxSharedLink;
import com.box.sdk.ProgressListener;
import com.box.sdk.ProgressOutputStream;
import com.box.sdk.RequestInterceptor;
import com.box.sdk.SharedLinkAPIConnection;
import com.box.sdk.StandardCharsets;
import com.box.sdk.Time;
import com.box.sdk.http.HttpMethod;
import com.box.sdk.internal.utils.CollectionUtils;
import com.eclipsesource.json.Json;
import com.eclipsesource.json.JsonObject;
import com.eclipsesource.json.ParseException;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import okhttp3.MediaType;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class BoxAPIRequest {
    private static final BoxLogger LOGGER = BoxLogger.defaultLogger();
    private static final String ERROR_CREATING_REQUEST_BODY = "Error creating request body";
    private static final int BUFFER_SIZE = 8192;
    private final BoxAPIConnection api;
    private final List<RequestHeader> headers;
    private final String method;
    private URL url;
    private BackoffCounter backoffCounter;
    private int connectTimeout;
    private int readTimeout;
    private InputStream body;
    private long bodyLength;
    private boolean shouldAuthenticate;
    private boolean followRedirects = true;
    private final String mediaType;

    public BoxAPIRequest(BoxAPIConnection api, URL url, String method) {
        this(api, url, method, "application/x-www-form-urlencoded");
    }

    protected BoxAPIRequest(BoxAPIConnection api, URL url, String method, String mediaType) {
        this.api = api;
        this.url = url;
        this.method = method;
        this.mediaType = mediaType;
        this.headers = new ArrayList<RequestHeader>();
        if (api != null) {
            Map<String, String> customHeaders = api.getHeaders();
            if (customHeaders != null) {
                for (String header : customHeaders.keySet()) {
                    this.addHeader(header, customHeaders.get(header));
                }
            }
            this.headers.add(new RequestHeader("X-Box-UA", api.getBoxUAHeader()));
        }
        this.backoffCounter = new BackoffCounter(new Time());
        this.shouldAuthenticate = true;
        if (api != null) {
            this.connectTimeout = api.getConnectTimeout();
            this.readTimeout = api.getReadTimeout();
        } else {
            this.connectTimeout = BoxGlobalSettings.getConnectTimeout();
            this.readTimeout = BoxGlobalSettings.getReadTimeout();
        }
        this.addHeader("Accept-Charset", "utf-8");
    }

    public BoxAPIRequest(BoxAPIConnection api, URL url, HttpMethod method) {
        this(api, url, method.name());
    }

    public BoxAPIRequest(URL url, HttpMethod method) {
        this(null, url, method.name());
    }

    public static boolean isRequestRetryable(BoxAPIException apiException) {
        return Objects.equals(apiException.getMessage(), ERROR_CREATING_REQUEST_BODY);
    }

    public static boolean isResponseRetryable(int responseCode, BoxAPIException apiException) {
        if (responseCode >= 500 || responseCode == 429) {
            return true;
        }
        return BoxAPIRequest.isClockSkewError(responseCode, apiException);
    }

    private static boolean isClockSkewError(int responseCode, BoxAPIException apiException) {
        String response = apiException.getResponse();
        if (response == null || response.length() == 0) {
            return false;
        }
        String message = apiException.getMessage();
        String errorCode = "";
        try {
            JsonObject responseBody = Json.parse((String)response).asObject();
            if (responseBody.get("code") != null) {
                errorCode = responseBody.get("code").toString();
            } else if (responseBody.get("error") != null) {
                errorCode = responseBody.get("error").toString();
            }
            return responseCode == 400 && errorCode.contains("invalid_grant") && message.contains("exp");
        }
        catch (ParseException e) {
            throw new BoxAPIException("API returned an error", responseCode, response);
        }
    }

    public void addHeader(String key, String value) {
        if (key.equals("As-User")) {
            for (int i = 0; i < this.headers.size(); ++i) {
                if (!this.headers.get(i).getKey().equals("As-User")) continue;
                this.headers.remove(i);
            }
        }
        if (key.equals("X-Box-UA")) {
            throw new IllegalArgumentException("Altering the X-Box-UA header is not permitted");
        }
        this.headers.add(new RequestHeader(key, value));
    }

    public int getConnectTimeout() {
        return this.connectTimeout;
    }

    public void setConnectTimeout(int timeout) {
        this.connectTimeout = timeout;
    }

    public int getReadTimeout() {
        return this.readTimeout;
    }

    public void setReadTimeout(int timeout) {
        this.readTimeout = timeout;
    }

    public void setFollowRedirects(boolean followRedirects) {
        this.followRedirects = followRedirects;
    }

    public InputStream getBody() {
        return this.body;
    }

    public void setBody(InputStream stream) {
        this.body = stream;
    }

    public void setBody(String body) {
        byte[] bytes = body.getBytes(StandardCharsets.UTF_8);
        this.bodyLength = bytes.length;
        this.body = new ByteArrayInputStream(bytes);
    }

    public void setBody(InputStream stream, long length) {
        this.bodyLength = length;
        this.body = stream;
    }

    public URL getUrl() {
        return this.url;
    }

    public void setUrl(URL url) {
        this.url = url;
    }

    public String getMethod() {
        return this.method;
    }

    List<RequestHeader> getHeaders() {
        return this.headers;
    }

    public BoxAPIResponse sendWithoutRetry() {
        return this.trySend(null);
    }

    public BoxAPIResponse send() {
        return this.send(null);
    }

    public BoxAPIResponse send(ProgressListener listener) {
        if (this.api == null) {
            this.backoffCounter.reset(BoxGlobalSettings.getMaxRetryAttempts() + 1);
        } else {
            this.backoffCounter.reset(this.api.getMaxRetryAttempts() + 1);
        }
        while (this.backoffCounter.getAttemptsRemaining() > 0) {
            try {
                return this.trySend(listener);
            }
            catch (BoxAPIException apiException) {
                if (!this.backoffCounter.decrement() || !BoxAPIRequest.isRequestRetryable(apiException) && !BoxAPIRequest.isResponseRetryable(apiException.getResponseCode(), apiException)) {
                    throw apiException;
                }
                LOGGER.warn(String.format("Retrying request due to transient error status=%d body=%s headers=%s", apiException.getResponseCode(), apiException.getResponse(), CollectionUtils.mapToString(apiException.getHeaders())));
                try {
                    this.resetBody();
                }
                catch (IOException ioException) {
                    throw apiException;
                }
                try {
                    List<String> retryAfterHeader = apiException.getHeaders().get("Retry-After");
                    if (retryAfterHeader == null) {
                        this.backoffCounter.waitBackoff();
                        continue;
                    }
                    int retryAfterDelay = Integer.parseInt(retryAfterHeader.get(0)) * 1000;
                    this.backoffCounter.waitBackoff(retryAfterDelay);
                }
                catch (InterruptedException interruptedException) {
                    Thread.currentThread().interrupt();
                    throw apiException;
                }
            }
        }
        throw new RuntimeException();
    }

    public void shouldAuthenticate(boolean shouldAuthenticate) {
        this.shouldAuthenticate = shouldAuthenticate;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    BoxFileUploadSessionPart sendForUploadPart(BoxFileUploadSession session, long offset) {
        if (this.api == null) {
            this.backoffCounter.reset(BoxGlobalSettings.getMaxRetryAttempts() + 1);
        } else {
            this.backoffCounter.reset(this.api.getMaxRetryAttempts() + 1);
        }
        while (this.backoffCounter.getAttemptsRemaining() > 0) {
            try (BoxJSONResponse response = (BoxJSONResponse)this.trySend(null);){
                JsonObject jsonObject = Json.parse((String)response.getJSON()).asObject();
                BoxFileUploadSessionPart boxFileUploadSessionPart = new BoxFileUploadSessionPart((JsonObject)jsonObject.get("part"));
                return boxFileUploadSessionPart;
            }
            catch (BoxAPIException apiException) {
                if (!this.backoffCounter.decrement()) throw apiException;
                if (!BoxAPIRequest.isRequestRetryable(apiException) && !BoxAPIRequest.isResponseRetryable(apiException.getResponseCode(), apiException)) {
                    throw apiException;
                }
                if (apiException.getResponseCode() == 500) {
                    try {
                        Iterable<BoxFileUploadSessionPart> parts = session.listParts();
                        for (BoxFileUploadSessionPart part : parts) {
                            if (part.getOffset() != offset) continue;
                            return part;
                        }
                    }
                    catch (BoxAPIException parts) {
                        // empty catch block
                    }
                }
                LOGGER.warn(String.format("Retrying request due to transient error status=%d body=%s", apiException.getResponseCode(), apiException.getResponse()));
                try {
                    this.resetBody();
                }
                catch (IOException ioException) {
                    throw apiException;
                }
                try {
                    this.backoffCounter.waitBackoff();
                }
                catch (InterruptedException interruptedException) {
                    Thread.currentThread().interrupt();
                    throw apiException;
                }
            }
        }
        throw new RuntimeException();
    }

    public String toString() {
        String lineSeparator = System.getProperty("line.separator");
        StringBuilder builder = new StringBuilder();
        builder.append("Request");
        builder.append(lineSeparator);
        builder.append(this.method);
        builder.append(' ');
        builder.append(this.url.toString());
        builder.append(lineSeparator);
        String bodyString = this.bodyToString();
        if (bodyString != null) {
            builder.append(lineSeparator);
            builder.append(bodyString);
        }
        return builder.toString().trim();
    }

    protected String bodyToString() {
        return null;
    }

    private void writeWithBuffer(OutputStream output, ProgressListener listener) {
        try {
            OutputStream finalOutput = output;
            if (listener != null) {
                finalOutput = new ProgressOutputStream(output, listener, this.bodyLength);
            }
            byte[] buffer = new byte[8192];
            int b = this.body.read(buffer);
            while (b != -1) {
                finalOutput.write(buffer, 0, b);
                b = this.body.read(buffer);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Error writting body", e);
        }
    }

    protected void resetBody() throws IOException {
        if (this.body != null) {
            this.body.reset();
        }
    }

    void setBackoffCounter(BackoffCounter counter) {
        this.backoffCounter = counter;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private BoxAPIResponse trySend(ProgressListener listener) {
        BoxAPIResponse response;
        RequestInterceptor interceptor;
        if (this.api != null && (interceptor = this.api.getRequestInterceptor()) != null && (response = interceptor.onRequest(this)) != null) {
            return response;
        }
        Request.Builder requestBuilder = new Request.Builder().url(this.getUrl());
        if (this.shouldAuthenticate) {
            requestBuilder.addHeader("Authorization", "Bearer " + this.api.lockAccessToken());
        }
        requestBuilder.addHeader("User-Agent", this.api.getUserAgent());
        requestBuilder.addHeader("X-Box-UA", this.api.getBoxUAHeader());
        this.headers.forEach(h -> {
            requestBuilder.removeHeader(h.getKey());
            requestBuilder.addHeader(h.getKey(), h.getValue());
        });
        if (this.api instanceof SharedLinkAPIConnection) {
            SharedLinkAPIConnection sharedItemAPI = (SharedLinkAPIConnection)this.api;
            String boxAPIValue = BoxSharedLink.getSharedLinkHeaderValue(sharedItemAPI.getSharedLink(), sharedItemAPI.getSharedLinkPassword());
            requestBuilder.addHeader("BoxApi", boxAPIValue);
        }
        try {
            long start = System.currentTimeMillis();
            this.writeMethodWithBody(requestBuilder, listener);
            Request request = requestBuilder.build();
            Response response2 = this.followRedirects ? this.api.execute(request) : this.api.executeWithoutRedirect(request);
            this.logDebug(String.format("[trySend] connection.connect() took %dms%n", System.currentTimeMillis() - start));
            BoxAPIResponse result = BoxAPIResponse.toBoxResponse(response2);
            this.logRequest();
            long getResponseStart = System.currentTimeMillis();
            this.logDebug(String.format("[trySend] Get Response (read network) took %dms%n", System.currentTimeMillis() - getResponseStart));
            BoxAPIResponse boxAPIResponse = result;
            return boxAPIResponse;
        }
        finally {
            if (this.shouldAuthenticate) {
                this.api.unlockAccessToken();
            }
        }
    }

    protected void writeMethodWithBody(Request.Builder requestBuilder, ProgressListener listener) {
        ByteArrayOutputStream bodyBytes = new ByteArrayOutputStream();
        if (this.body != null) {
            long writeStart = System.currentTimeMillis();
            this.writeWithBuffer(bodyBytes, listener);
            this.logDebug(String.format("[trySend] Body write took %dms%n", System.currentTimeMillis() - writeStart));
        }
        if (this.method.equals("GET")) {
            requestBuilder.get();
        }
        if (this.method.equals("DELETE")) {
            requestBuilder.delete();
        }
        if (this.method.equals("OPTIONS")) {
            if (this.body == null) {
                requestBuilder.method("OPTIONS", null);
            } else {
                requestBuilder.method("OPTIONS", RequestBody.create((byte[])bodyBytes.toByteArray(), (MediaType)this.mediaType()));
            }
        }
        if (this.method.equals("POST")) {
            requestBuilder.post(RequestBody.create((byte[])bodyBytes.toByteArray(), (MediaType)this.mediaType()));
        }
        if (this.method.equals("PUT")) {
            requestBuilder.put(RequestBody.create((byte[])bodyBytes.toByteArray(), (MediaType)this.mediaType()));
        }
    }

    private void logDebug(String message) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug(message);
        }
    }

    private void logRequest() {
        this.logDebug(this.toString());
    }

    protected MediaType mediaType() {
        return MediaType.parse((String)this.mediaType);
    }

    public static final class RequestHeader {
        private final String key;
        private final String value;

        public RequestHeader(String key, String value) {
            this.key = key;
            this.value = value;
        }

        public String getKey() {
            return this.key;
        }

        public String getValue() {
            return this.value;
        }
    }
}

