/*
 * Copyright (c) BrowserStack, Inc. Proprietary content. All rights reserved. Unauthorized use,
 * copies, and derivatives of this file via any medium is strictly prohibited.
 */

package com.browserstack.automate.testassist;

import static com.browserstack.automate.testassist.TestSessionRecorder.DEBUG;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.RemoteWebDriver;
import java.io.PrintStream;
import java.util.LinkedList;

@Aspect
public class AutomateAspects {
  private static final String TEST_FORMAT_HASH = "hash";
  private static final String TEST_FORMAT_NAME = "name";
  private static final String TEST_FORMAT =
      System.getProperty("browserstack.automate.test.format", TEST_FORMAT_HASH);
  private static final String OUTPUT_FORMAT = "browserstack:session:%s:test:%s";

  private TestSessionRecorder recorder = TestSessionRecorder.getInstance();
  private PrintStream outStream = System.out;


  @Pointcut("execution(public org.openqa.selenium.remote.RemoteWebDriver.new(..))")
  public void webDriverConstructor() {}

  @Pointcut("execution(public void org.openqa.selenium.remote.RemoteWebDriver.quit())")
  public void webDriverQuit() {}

  @Pointcut("execution(@org.junit.Test * *(..))")
  public void junitTestCase() {}

  @Pointcut("execution(@org.testng.annotations.Test * *(..))")
  public void testngTestCase() {}


  @Before("junitTestCase() || testngTestCase()")
  public void beforeTestCase(JoinPoint joinPoint) {
    TestSessionRecorder.getInstance().recordClassHierarchy(joinPoint.getTarget());
    String testSignature = TestSessionRecorder.getSignature(joinPoint);
    UnitTestCase unitTestCase =
        new UnitTestCase(joinPoint, testSignature, recorder.getTestCaseIndex(joinPoint));

    recorder.getCapturedTestCases().add(unitTestCase);
    if (DEBUG)
      System.out.println(">>> ADD TC: " + unitTestCase.data);

    recorder.setLastEventType(EventType.TEST_START);
  }

  @After("junitTestCase() || testngTestCase()")
  public void afterTestCase(JoinPoint joinPoint) {
    String testSignature = TestSessionRecorder.getSignature(joinPoint);
    UnitTestCase tc =
        (UnitTestCase) recorder.findAndRemoveEvent(testSignature, recorder.getCapturedTestCases());

    if (tc != null) {
      if (DEBUG)
        System.out.println(">>> REMOVE TC: " + tc.data);

      String sessionId;
      if (recorder.getLastEventType() == EventType.WD_END) {
        // WebDriver has already been dequeued. Use last sessionId.
        sessionId = recorder.getLastSessionId();
      } else {
        LinkedList<WebDriverInstance> capturedWebDrivers = recorder.getCapturedWebDrivers();
        if (DEBUG)
          System.out.println(">>> REMOVE TC: capturedWebDrivers: " + capturedWebDrivers.size()
              + " | Thread: " + Thread.currentThread().getId());
        if (!capturedWebDrivers.isEmpty()) {
          sessionId = capturedWebDrivers.getLast().data;
        } else {
          // last WebDriver instance across threads
          sessionId =
              TestSessionRecorder.getInstance().findSessionIdForClass(joinPoint.getTarget());
        }
      }

      if (sessionId != null) {
        if (DEBUG)
          System.out.println(">>> REMOVE TC: sessionId: " + sessionId);
        recorder.linkSessionIdToTestCase(sessionId, tc);
        String sessionInfo = String.format(OUTPUT_FORMAT, sessionId,
            TEST_FORMAT.equals(TEST_FORMAT_NAME) ? tc.getUniqueName() : tc.getTestHash());
        outStream.println(sessionInfo);
        recorder.setLastEventType(EventType.TEST_END);
      }
    }
  }

  @Before("webDriverConstructor()")
  public void beforeNewWebDriver(JoinPoint joinPoint) {
    Object target = joinPoint.getTarget();

    if (target instanceof RemoteWebDriver
        && TestSessionRecorder.isBrowserStackWebDriver(joinPoint)) {
      Object[] constructorArgs = joinPoint.getArgs();

      if (constructorArgs.length > 1 && constructorArgs[1] instanceof DesiredCapabilities) {
        TestSessionRecorder.setBuild((DesiredCapabilities) constructorArgs[1]);
      }
    }
  }

  @After("webDriverConstructor()")
  public void afterNewWebDriver(JoinPoint joinPoint) {
    Object target = joinPoint.getTarget();

    if (target instanceof RemoteWebDriver) {
      RemoteWebDriver webDriver = (RemoteWebDriver) target;
      WebDriverInstance webDriverInstance = new WebDriverInstance(webDriver);

      if (!TestSessionRecorder.isBrowserStackWebDriver(joinPoint)
          || recorder.isWebDriverCaptured(webDriverInstance)) {
        return;
      }

      recorder.captureWebDriver(webDriverInstance);
      recorder.setLastEventType(EventType.WD_START);
      TestSessionRecorder.getInstance().registerSessionIdForCurrentClass(webDriverInstance.data);
      if (DEBUG)
        outStream.println(">>> ADD WD: " + webDriverInstance.data + " | "
            + recorder.getCapturedWebDrivers().size() + " | Thread: "
            + Thread.currentThread().getId());
    }
  }

  @After("webDriverQuit()")
  public void afterWebDriverQuit(JoinPoint joinPoint) {
    Object target = joinPoint.getTarget();

    if (target instanceof RemoteWebDriver) {
      RemoteWebDriver webDriver = (RemoteWebDriver) target;
      String sessionId =
          webDriver.getSessionId() != null ? webDriver.getSessionId().toString() : null;

      if (sessionId != null) {
        RecordedData webDriverInstance =
            recorder.findAndRemoveEvent(sessionId, recorder.getCapturedWebDrivers());
        if (webDriverInstance != null) {
          if (DEBUG)
            System.out.println(">>> REMOVE WD: " + webDriverInstance.data);

          recorder.setLastEventType(EventType.WD_END);
          recorder.setLastSessionId(webDriverInstance.data);
        }
      }
    }
  }
}
