package com.browserstack.automate.testassist;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import static com.browserstack.automate.testassist.TestSessionRecorder.DEBUG;

/**
 * @author Shirish Kamath
 * @author Anirudha Khanna
 */
public class XmlReporter {

  private static final String DIR_REPORTS = "browserstack-reports";
  private static final String REPORT_FILE_PREFIX = "REPORT-";
  private static final String REPORT_FILE_EXTN = ".xml";
  private static final String DIR_TARGET = "target";

  private final Map<String, Document> reports;

  private final File targetDir;

  private final boolean isEnabled;

  public XmlReporter() {
    this(System.getProperty("user.dir", ".") + File.separator + DIR_TARGET);
  }

  public XmlReporter(String outputPath) {
    this.targetDir = new File(outputPath);
    this.isEnabled = (targetDir.exists() && targetDir.isDirectory() && targetDir.canWrite());
    this.reports = new ConcurrentHashMap<String, Document>();
  }

  public synchronized void saveXML() throws Exception {
    if (!isEnabled) {
      System.out.println(
          "[ERROR] Target directory missing or not writable: " + targetDir.getAbsolutePath());
      return;
    }

    for (Map.Entry<String, Document> entry : reports.entrySet()) {
      Document document = entry.getValue();
      if (document == null) {
        continue;
      }

      FileWriter fileWriter = null;

      try {
        File destDir = new File(targetDir, DIR_REPORTS);
        FileUtils.forceMkdir(destDir);
        fileWriter = new FileWriter(
            new File(destDir, REPORT_FILE_PREFIX + entry.getKey() + REPORT_FILE_EXTN));

        Transformer transformer = TransformerFactory.newInstance().newTransformer();
        transformer.setOutputProperty(OutputKeys.INDENT, "yes");
        DOMSource source = new DOMSource(document);
        StreamResult result = new StreamResult(fileWriter);
        transformer.transform(source, result);
      } catch (IOException e) {
        // ignore?
        if (DEBUG) {
          System.out.println("[ERROR] Failed to save report for: " + entry.getKey());
          e.printStackTrace();
        }
      } finally {
        IOUtils.closeQuietly(fileWriter);
      }
    }
  }

  public void appendTest(final UnitTestCase testCase, final String sessionId,
      final String projectType) throws Exception {
    if (!isEnabled) {
      return;
    }

    Document document = reports.get(testCase.getFullClassName());
    if (document == null) {
      document = initXML(testCase);

      if (document != null) {
        reports.put(testCase.getFullClassName(), document);
      }
    }

    if (document != null) {
      Element testsElement = document.getDocumentElement();
      if (testsElement != null) {
        Element test = document.createElement("testcase");
        test.setAttribute("name", testCase.testName);
        test.setAttribute("classname", testCase.getFullClassName());

        // additional attributes
        test.setAttribute("id", testCase.getUniqueName());
        test.setAttribute("index", "" + testCase.testCaseIndex);
        test.setAttribute("package", "" + testCase.packageName);
        test.setAttribute("class", "" + testCase.className);

        Element sessionNode = document.createElement("session");
        sessionNode.setTextContent(sessionId);
        test.appendChild(sessionNode);

        Element projectTypeNode = document.createElement("projectType");
        projectTypeNode.setTextContent(projectType);
        test.appendChild(projectTypeNode);

        testsElement.appendChild(test);
      }
    }
  }

  private static Document initXML(final UnitTestCase testCase) {
    DocumentBuilder docBuilder;
    try {
      docBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
    } catch (ParserConfigurationException e) {
      return null;
    }

    Document doc = docBuilder.newDocument();
    Element testsElement = doc.createElement("testsuite");
    testsElement.setAttribute("name", testCase.getFullClassName());
    doc.appendChild(testsElement);
    return doc;
  }
}
