/**
 * Copyright 2013-2019 Butor Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.butor.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.lang.reflect.Field;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;
import java.util.regex.Pattern;

import org.butor.json.CommonRequestArgs;
import org.butor.json.JsonHelper;
import org.butor.json.service.Context;
import org.butor.json.service.ResponseHandler;
import org.springframework.jdbc.BadSqlGrammarException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.test.jdbc.JdbcTestUtils;

import com.google.common.base.Throwables;
import com.google.common.io.Closeables;

public class ButorTestUtil {
	public Context context(final CommonRequestArgs commonRequestArgs, final ResponseHandler<Object> rh) {
		return new Context() {
			
			@Override
			public ResponseHandler<Object> getResponseHandler() {
				return rh;
			}
			
			@Override
			public CommonRequestArgs getRequest() {
				return commonRequestArgs;
			}
		};
	}
	public static CommonRequestArgs getCommonRequestArgs(String commonRequestJSON) {
		CommonRequestArgs commonRequestArgs = new JsonHelper().deserialize(commonRequestJSON, CommonRequestArgs.class);
		return commonRequestArgs;
	}
	public  void assertObjectEqual(Object expected, Object actual) {
		assertObjectEqual("",expected,actual);
	}
	
	
	public  void assertObjectEqual(String message,Object expected, Object actual) {
		Map<String, Object> mapExpected =  getFieldMap(expected);
		Map<String, Object> mapActualMap =  getFieldMap(actual);
		
		for (Entry<String,Object> entry : mapExpected.entrySet()) {
			Object actualValue = mapActualMap.get(entry.getKey());
			Object expectedValue = entry.getValue();
			
			assertEquals(String.format("Field differs : %s, %s\n",entry.getKey(),message),expectedValue, actualValue);
		
		}
		assertEquals(expected, actual);
		
	}
	
	private  Map<String, Object> getFieldMap(Object o1) {
		Map<String, Object> mapObject1 = new TreeMap<String, Object>();
		for (Field f : o1.getClass().getDeclaredFields()) {
			f.setAccessible(true);
			try {
				mapObject1.put(f.getName(), f.get(o1));
			} catch (Exception e) {
				Throwables.propagate(e);
			}
		}
		return mapObject1;
	}

	public void executeSQL(JdbcTemplate jt, URL sqlURL, String dbName) throws IOException {
		executeSQL(jt, sqlURL, dbName, "derby");
	}

	public void executeSQL(JdbcTemplate jt, URL sqlURL, String dbName, String jdbcDriverType) throws IOException {
		String sql = getStringFromURL(sqlURL);
		executeSQL(jt, sql, dbName, jdbcDriverType);
	}
	
	public String getStringFromURL(URL url) throws IOException {
		StringWriter sw = new StringWriter();
		InputStream is =null;
		try {
			is = url.openStream();
			while(true) {
				int c = is.read();
				if (c == -1)
					break;
				sw.append((char)c);
			}
		} finally {
			Closeables.closeQuietly(is);
		}
		return sw.toString();
	}
	
	
	public void executeSQL(JdbcTemplate jt, String sqlData, String dbName, String jdbcDriverType) throws IOException {
		assertNotNull(jt);
		
		List<String> scriptlines = new ArrayList<String>();
		// TODO extract this code from JdbcTestUtils so that we don't
		// include spring-test in compile scope if we move the method to butor-dao?
		JdbcTestUtils.splitSqlScript(sqlData, ';', scriptlines);
		for (String sqlOri : scriptlines) {
			try {
				String trimmedSql = sqlOri.trim();
				if (trimmedSql.length()==0) {
					continue;
				}
				if (jdbcDriverType.equals("derby")) {
					if (!trimmedSql.startsWith("DROP DATABASE") && !trimmedSql.startsWith("ALTER TABLE")) {
						String sql = sqlOri.replace("`", "");
						sql = sql.replace("CREATE DATABASE IF NOT EXISTS " + dbName, "CREATE SCHEMA " + dbName);
						sql = sql.replace("DROP TABLE IF EXISTS ", "DROP TABLE ");
						sql = sql.replaceAll("comment '[^']*'", "");
						sql = sql.replaceAll("unsigned", "");
						sql = sql.replaceAll("TEXT", "CLOB");
						
						// temporary replace of 'NOT NULL' to avoid collision with NULL replace
						sql = sql.replaceAll("(?i)NOT NULL", "NOT_NU__");
						// replace first null in table definition.
						sql = sql.replace("NULL", "");
						
						sql = sql.replaceAll("(?i)DATETIME", "TIMESTAMP");
						sql = sql.replaceAll("(?i)NOW\\(\\)", "CURRENT_TIMESTAMP");
						sql = sql.replaceAll("(?i)now", "CURRENT_TIMESTAMP");
						sql = sql.replaceAll("NOT_NU__", "NOT NULL");
						sql = sql.replaceAll("(?i)ENGINE = InnoDB", "");
						sql = sql.replaceAll("(?i)AUTO_INCREMENT", "GENERATED BY DEFAULT AS IDENTITY");
						System.out.println(sql);
						jt.execute(sql);
					}
				}
				else {
					System.out.println(sqlOri);
					jt.execute(sqlOri);
				}
			} catch (Throwable e) {
				System.out.println("====> " + e.getMessage());
				if (e instanceof BadSqlGrammarException) {
					String msg = ((BadSqlGrammarException) e).getSQLException().getMessage();
					Pattern ptr = Pattern.compile("^Table .* not found.*");
					if (msg.indexOf("because it does not exist") > -1 ||
							ptr.matcher(msg).find() ||
							msg.indexOf("object not found") > -1) {
						continue;
					} else if (msg.indexOf("already exists") > -1) {
						continue;
					}
					fail(e.getMessage());
				}
			}
		}
	}
	public static JsonHelper getJsonHelperWithLocalTimeFormatter() {
		return new JsonHelper("yyyy-MM-dd HH:mm:ss.SSS");
	}
}
