/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.api.compat.v3_4

import com.android.build.gradle.api.BaseVariant
import com.android.build.gradle.internal.pipeline.TransformTask
import com.android.build.gradle.internal.res.LinkApplicationAndroidResourcesTask
import com.android.repository.Revision
import com.bytedance.ultimate.inflater.plugin.api.compat.ConfigFileCollection
import com.bytedance.ultimate.inflater.plugin.api.compat.IUltimateInflaterPluginAPICompat
import com.bytedance.ultimate.inflater.plugin.api.compat.capitalizeUS
import com.google.auto.service.AutoService
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.file.FileCollection
import java.io.File
import java.util.concurrent.atomic.AtomicBoolean

/**
 * Created by ChengTao(chentao.joe@bytedance.com) on 2023/3/2.
 */
@AutoService(IUltimateInflaterPluginAPICompat.Factory::class)
class V34UltimateInflaterPluginAPICompatFactory : IUltimateInflaterPluginAPICompat.Factory {

    override val revision: Revision
        get() = Revision(3, 4)

    override fun create(): IUltimateInflaterPluginAPICompat {
        return Impl()
    }

    class Impl : IUltimateInflaterPluginAPICompat {
        override fun processResourcesTask(project: Project, variant: BaseVariant): Task {
            return project.tasks.findByName("process${variant.name.capitalizeUS()}Resources")
                ?: throw IllegalArgumentException("cannot find task with processResources")
        }

        override fun processResourcesTaskOutputDir(project: Project, variant: BaseVariant): File {

            return processResourcesTask(project, variant)
                .let { it as? LinkApplicationAndroidResourcesTask }
                ?.getResPackageOutputFolder()
                ?: throw IllegalArgumentException("cannot get outputDir for task processResources")
        }

        override fun resolveConfigFile(
            project: Project,
            variant: BaseVariant
        ): ConfigFileCollection {
            val task = project.tasks
                .getByName("transformResourcesWithMergeJavaResFor${variant.name.capitalizeUS()}")
                .let { it as TransformTask }
            return LazyConfigFileCollection(project, task.inputs.files)
        }

        class LazyConfigFileCollection(
            private val project: Project,
            private val inputs: FileCollection
        ) : ConfigFileCollection() {

            private val resolved = AtomicBoolean(false)

            private lateinit var _subProjectConfigFiles: FileCollection

            private lateinit var _externalLibConfigFiles: FileCollection

            override val appProjectConfigFiles: FileCollection by lazy { project.files() }
            override val subProjectConfigFiles: FileCollection
                get() {
                    resolveConfigFiles()
                    return _subProjectConfigFiles
                }
            override val externalLibConfigFiles: FileCollection
                get() {
                    resolveConfigFiles()
                    return _externalLibConfigFiles
                }

            private fun resolveConfigFiles() {
                if (resolved.get().not()) {
                    synchronized(this) {
                        if (resolved.get().not()) {
                            val subProjectConfigFileList = mutableListOf<File>()
                            val externalLibConfigFileList = mutableListOf<File>()
                            inputs.filter { it.name.endsWith(".jar") }.forEach { file ->
                                if (file.name.contains("library_java_res")) {
                                    subProjectConfigFileList.add(file)
                                } else {
                                    externalLibConfigFileList.add(file)
                                }
                            }
                            _subProjectConfigFiles = project.files(subProjectConfigFileList)
                            _externalLibConfigFiles = project.files(externalLibConfigFileList)
                        }
                    }
                }
            }
        }

    }
}