/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.api.compat.v4_1

import com.android.build.gradle.api.BaseVariant
import com.android.build.gradle.internal.res.LinkApplicationAndroidResourcesTask
import com.android.build.gradle.internal.tasks.MergeJavaResourceTask
import com.android.repository.Revision
import com.bytedance.ultimate.inflater.plugin.api.compat.ConfigFileCollection
import com.bytedance.ultimate.inflater.plugin.api.compat.IUltimateInflaterPluginAPICompat
import com.bytedance.ultimate.inflater.plugin.api.compat.capitalizeUS
import com.google.auto.service.AutoService
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.file.FileCollection
import java.io.File

/**
 * Created by ChengTao(chentao.joe@bytedance.com) on 2023/3/2.
 */
@AutoService(IUltimateInflaterPluginAPICompat.Factory::class)
class V41UltimateInflaterPluginAPICompatFactory : IUltimateInflaterPluginAPICompat.Factory {

    override val revision: Revision
        get() = Revision(4, 1)

    override fun create(): IUltimateInflaterPluginAPICompat {
        return Impl()
    }

    class Impl : IUltimateInflaterPluginAPICompat {
        override fun processResourcesTask(project: Project, variant: BaseVariant): Task {
            return project.tasks.findByName("process${variant.name.capitalizeUS()}Resources")
                ?: throw IllegalArgumentException("cannot find task with processResources")
        }

        override fun processResourcesTaskOutputDir(project: Project, variant: BaseVariant): File {
            return processResourcesTask(project, variant)
                .let { it as? LinkApplicationAndroidResourcesTask }
                ?.resPackageOutputFolder?.orNull?.asFile
                ?: throw IllegalArgumentException("cannot get outputDir for task processResources")
        }

        override fun resolveConfigFile(
            project: Project,
            variant: BaseVariant
        ): ConfigFileCollection {
            val task = project.tasks.getByName(
                "merge${variant.name.capitalizeUS()}JavaResource"
            ) as MergeJavaResourceTask
            return object : ConfigFileCollection() {
                override val appProjectConfigFiles: FileCollection
                    get() = project.files()
                override val subProjectConfigFiles: FileCollection
                    get() = task.subProjectJavaRes ?: project.files()
                override val externalLibConfigFiles: FileCollection
                    get() = task.externalLibJavaRes ?: project.files()
            }
        }
    }

}
