/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.api.compat

import com.android.build.gradle.api.BaseVariant
import com.android.builder.model.Version
import com.android.repository.Revision
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.file.FileCollection
import java.io.File
import java.util.*

/**
 * Created by ChengTao(chentao.joe@bytedance.com) on 2023/6/25.
 */
interface IUltimateInflaterPluginAPICompat {
    /**
     * get processResources Task
     *
     * @param project the [Project]
     * @param variant the [BaseVariant]
     *
     * @return the processResources Task for the given [variant]
     */
    fun processResourcesTask(project: Project, variant: BaseVariant): Task

    /**
     * get the output file of processResources Task
     *
     * @param project the [Project]
     * @param variant the [BaseVariant]
     *
     * @return the output file of processResources Task, must be directory.
     */
    fun processResourcesTaskOutputDir(project: Project, variant: BaseVariant): File


    /**
     * Layout config file of UltimateInflater is located at
     * "resources/META-INF/ultimate-inflater/" in every module, jar or aar.
     *
     * Since UltimateInflater plugin generates source codes before transform phase, so collecting all
     * layout config file is not that easy. One way to resolve this is to merge all config file in
     * "mergeJavaResource" phase, but the task named "compileJavaWithJavac" will be executed before the
     * task "mergeJavaResource", so it's impossible to get output of "mergeJavaResource" before
     * "compileJavaWithJavac".
     *
     * resolve all layout config file crossing from multiple modules
     *
     * @param project the [Project]
     * @param variant the [BaseVariant]
     *
     * @return the resolved config files
     */
    fun resolveConfigFile(project: Project, variant: BaseVariant): ConfigFileCollection

    interface Factory {

        val revision: Revision

        fun create(): IUltimateInflaterPluginAPICompat
    }
}

object UltimateInflaterPluginAPICompat : IUltimateInflaterPluginAPICompat {

    private const val CONFIG_FILE_DIR = "src/main/resources/META-INF/ultimate-inflater"
    private val configFilePaths = listOf(
        "$CONFIG_FILE_DIR/layout-config.toml",
        "$CONFIG_FILE_DIR/view-config.toml",
        "$CONFIG_FILE_DIR/layout-inflater-factory-config.toml"
    )

    private val revision by lazy {
        Revision.parseRevision(Version.ANDROID_GRADLE_PLUGIN_VERSION)
    }

    private val factories by lazy {
        ServiceLoader.load(
            IUltimateInflaterPluginAPICompat.Factory::class.java,
            IUltimateInflaterPluginAPICompat::class.java.classLoader
        ).sortedByDescending { it.revision }.toList().also { list ->
            if (list.isEmpty()) {
                throw IllegalArgumentException(
                    "Cannot find implementations of IUltimateInflaterPluginAPICompat.Factory"
                )
            }
        }
    }

    private val impl: IUltimateInflaterPluginAPICompat by lazy {
        val factory = factories.firstOrNull {
            it.revision.major == revision.major && it.revision.minor == revision.minor
        } ?: factories.firstOrNull {
            it.revision.minor == revision.minor
        } ?: factories.first()
        factory.create()
    }

    override fun processResourcesTask(project: Project, variant: BaseVariant): Task {
        return impl.processResourcesTask(project, variant)
    }

    override fun processResourcesTaskOutputDir(project: Project, variant: BaseVariant): File {
        return impl.processResourcesTaskOutputDir(project, variant)
    }

    override fun resolveConfigFile(project: Project, variant: BaseVariant): ConfigFileCollection {
        val configFile = impl.resolveConfigFile(project, variant)
        return object : ConfigFileCollection() {
            override val appProjectConfigFiles: FileCollection
                get() = configFile.appProjectConfigFiles.takeIf { it.isEmpty.not() }
                    ?: project.rootProject.files(
                        configFilePaths
                            .map { path -> File(project.projectDir, path.toSystemPath()) }
                            .filter { it.exists() }
                    )
            override val subProjectConfigFiles: FileCollection
                get() = configFile.subProjectConfigFiles
            override val externalLibConfigFiles: FileCollection
                get() = configFile.externalLibConfigFiles
        }
    }
}