/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin

import com.android.build.gradle.AppExtension
import com.bytedance.ultimate.inflater.plugin.api.compat.UltimateInflaterPluginAPICompat
import com.bytedance.ultimate.inflater.plugin.internal.Helper
import com.bytedance.ultimate.inflater.plugin.internal.capitalizeUS
import com.bytedance.ultimate.inflater.plugin.internal.config.ConfigManager
import com.bytedance.ultimate.inflater.plugin.internal.task.GenerateSourcesTask
import com.bytedance.ultimate.inflater.plugin.internal.transformer.LayoutCreatorFactoryRegistryTransformer
import com.bytedance.ultimate.inflater.plugin.internal.transformer.LayoutNameToIdRegistryTransformer
import com.bytedance.ultimate.inflater.plugin.internal.transformer.UltimateInflaterApplicationThemeTransformer
import com.bytedance.ultimate.inflater.plugin.internal.transformer.ViewCreatorTransformer
import com.bytedance.ultimate.inflater.plugin.internal.transformer.ViewHelperTransformer
import com.bytedance.ultimate.inflater.plugin.internal.transformer.XmlBlockHelperTransformer
import org.gradle.api.Project
import org.objectweb.asm.tree.ClassNode

/**
 * Created by chentao.joe on 2022/7/29
 * @author chentao.joe@bytedance.com
 */
object UltimateInflaterPluginAPI {

    @JvmStatic
    fun postOnApply(project: Project, android: AppExtension) {
        // 1. exclude UltimateInflater layout config file, we do not need packing it to apk.
        android.packagingOptions.apply {
            exclude("META-INF/ultimate-inflater/**")
        }
        // 2. register task
        project.afterEvaluate {
            ConfigManager.parseExtensionConfig(it)
            registerGenerateSourcesTask(it, android)
        }
    }

    private fun registerGenerateSourcesTask(project: Project, android: AppExtension) {
        android.applicationVariants.forEach { variant ->
            val processResourcesTask = UltimateInflaterPluginAPICompat
                .processResourcesTask(project, variant)
            val processedResourcesDir = UltimateInflaterPluginAPICompat
                .processResourcesTaskOutputDir(project, variant)
            val generatedSourceRoot = Helper.createGeneratedSourceRoot(project, variant)
            val configFileCollection = UltimateInflaterPluginAPICompat
                .resolveConfigFile(project, variant)
            val generateTask = project.tasks.create(
                "generate${variant.name.capitalizeUS()}UltimateInflaterSources",
                GenerateSourcesTask::class.java
            )
            generateTask.processResourcesDir = processedResourcesDir
            generateTask.generateJavaSourceRoot = generatedSourceRoot
            generateTask.configFiles = configFileCollection
            generateTask.dependsOn(processResourcesTask)
            variant.registerJavaGeneratingTask(generateTask, generatedSourceRoot)
        }
    }

    private val transformers by lazy {
        listOf(
            ViewHelperTransformer(),
            XmlBlockHelperTransformer(),
            LayoutNameToIdRegistryTransformer(),
            LayoutCreatorFactoryRegistryTransformer(),
            UltimateInflaterApplicationThemeTransformer(),
            ViewCreatorTransformer()
        )
    }

    @JvmStatic
    fun transformClass(klass: ClassNode) {
        transformers.forEach { transformer ->
            transformer.transform(klass)
        }
    }
}
