/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

import java.lang.reflect.Field
import java.lang.reflect.ParameterizedType

/**
 * Created by chentao.joe on 2022/7/19
 * @author chentao.joe@bytedance.com
 */
data class ResStringPoolHeader(
    @unit_size_t_group
    val header: ResChunkHeader,
    @unit_32_t
    val stringCount: Int,
    @unit_32_t
    val styleCount: Int,
    @unit_32_t
    @FieldCast(FlagsFieldCaster::class)
    val flags: Flags,
    @unit_32_t
    val stringsStart: Int,
    @unit_32_t
    val stylesStart: Int
) {

    enum class Flags(val value: Int) {
        UNKNOWN(0),

        SORTED_FLAG(1 shl 0),

        // String pool is encoded in UTF-8
        UTF8_FLAG(1 shl 8);

        fun isUtf8(): Boolean = this == UTF8_FLAG
    }

    class FlagsFieldCaster : IntFieldTransformCaster<Flags>() {
        override fun castTransform(value: Int) = value.toFlags()
        override fun unCastTransform(value: Flags) = value.value
    }
}

fun Int.toFlags(): ResStringPoolHeader.Flags {
    return ResStringPoolHeader.Flags.values().find { it.value == this }
        ?: ResStringPoolHeader.Flags.UNKNOWN
}

data class ResStringPool(
    @unit_size_t_group
    val header: ResStringPoolHeader,
    @FieldCast(FieldEntriesCaster::class)
    val entries: List<Int>,
    @FieldCast(FieldEntryStylesCaster::class)
    val entryStyles: List<Int>,
    @FieldCast(FieldStringsCaster::class)
    val strings: List<StringInfo>,
    @FieldCast(FieldStylesCaster::class)
    val styles: List<StyleInfo>
) {

    init {
        styles.getAllResStringPoolRefs()?.forEach { ref ->
            ref.updateValue(this)
        }
    }

    class FieldEntriesCaster : AbsFieldCaster<List<Int>>() {

        private val header: ResStringPoolHeader by lazy { getFieldValue<ResStringPoolHeader>("header") }

        override fun getFieldSize(field: Field, value: List<Int>): Int {
            return value.size * 4
        }

        override fun cast(field: Field, byteList: ByteList): List<Int> {
            return header.stringCount
                .takeIf { it > 0 }
                ?.let { byteList.subList(0, header.stringCount * 4).chunked(4).map { it.toInt() } }
                ?: emptyList()
        }

        override fun unCast(
            field: Field,
            value: List<Int>,
            interceptor: UnCastInterceptor
        ): List<Byte> {
            return interceptor.intercept(field, value)
                ?: throw IllegalStateException("Cannot unCast entries")
        }
    }

    class FieldEntryStylesCaster : AbsFieldCaster<List<Int>>() {

        private val header: ResStringPoolHeader by lazy { getFieldValue<ResStringPoolHeader>("header") }

        override fun getFieldSize(field: Field, value: List<Int>): Int {
            return value.size * 4
        }

        override fun cast(field: Field, byteList: ByteList): List<Int> {
            return header.styleCount
                .takeIf { it > 0 }
                ?.let { byteList.subList(0, header.styleCount * 4).chunked(4).map { it.toInt() } }
                ?: emptyList()
        }

        override fun unCast(
            field: Field,
            value: List<Int>,
            interceptor: UnCastInterceptor
        ): List<Byte> {
            return interceptor.intercept(field, value)
                ?: throw IllegalStateException("Cannot unCast entryStyle")
        }
    }

    class FieldStringsCaster : AbsFieldCaster<List<StringInfo>>() {

        private val header: ResStringPoolHeader by lazy { getFieldValue<ResStringPoolHeader>("header") }
        private val entries: List<Int> by lazy { getFieldValue<List<Int>>("entries") }

        override fun getFieldSize(field: Field, value: List<StringInfo>): Int {
            return if (header.styleCount > 0) {
                header.stylesStart - header.stringsStart
            } else {
                header.header.size - header.stringsStart
            }
        }

        override fun cast(field: Field, byteList: ByteList): List<StringInfo> {
            return entries.takeIf { it.isNotEmpty() }?.zipWithNext()
                ?.plus(listOf(entries.last() to (header.header.size - header.stringsStart)))
                ?.map {
                    StringInfo.parse(
                        ByteList(byteList.subList(it.first, it.second)),
                        header.flags.isUtf8()
                    )
                }
                ?: emptyList()
        }

        override fun unCast(
            field: Field,
            value: List<StringInfo>,
            interceptor: UnCastInterceptor
        ): List<Byte> {
            return interceptor.intercept(field, value)
                ?: throw IllegalStateException("Cannot unCast strings")
        }
    }

    class FieldStylesCaster : AbsFieldCaster<List<StyleInfo>>() {

        private val header: ResStringPoolHeader by lazy { getFieldValue<ResStringPoolHeader>("header") }
        private val entryStyles: List<Int> by lazy { getFieldValue<List<Int>>("entryStyles") }
        override fun cast(field: Field, byteList: ByteList): List<StyleInfo> {
            return entryStyles.takeIf { it.isNotEmpty() }?.asSequence()?.zipWithNext()
                ?.plus(listOf(entryStyles.last() to header.header.size - header.stylesStart))
                ?.map { byteList.subList(it.first, it.second) }
                ?.map { it to ByteList(it) }
                ?.map { (styleData, styleByeList) ->
                    val spanList = mutableListOf<ResStringPoolSpan>()
                    val spanSize = sizeOf(ResStringPoolSpan::class)
                    while (styleByeList.isValid && StyleInfo.isEnd(styleByeList)) {
                        spanList.add(cast(styleByeList))
                        styleByeList += spanSize
                    }
                    StyleInfo(styleData, spanList)
                }?.toList()
                ?: emptyList()
        }

        override fun unCast(
            field: Field,
            value: List<StyleInfo>,
            interceptor: UnCastInterceptor
        ): List<Byte> {
            return interceptor.intercept(field, value)
                ?: throw IllegalStateException("Cannot unCast styles")
        }
    }

    data class StringInfo(
        val data: List<Byte>,
        val size: Int,
        val string: String
    ) {
        companion object {
            fun parse(byteList: ByteList, isUtf8: Boolean): StringInfo {
                return if (isUtf8) {
                    parseUtf8String(byteList)
                } else {
                    parseUtf16String(byteList)
                }
            }

            private fun parseUtf8String(byteList: ByteList): StringInfo {
                val data = byteList.subList(0)
                decodeLength(byteList)
                val len = decodeLength(byteList)
                val string = String(byteList.subList(0, len).toByteArray())
                return StringInfo(data, len, string)
            }

            private fun parseUtf16String(byteList: ByteList): StringInfo {
                val data = byteList.subList(0)
                val len = decodeLength(byteList)
                val string = String(byteList.subList(0, len * 2).toByteArray(), Charsets.UTF_16)
                return StringInfo(data, len, string)
            }

            private fun decodeLength(byteList: ByteList): Int {
                var len = byteList.castToInt(0, 1)
                if ((len and 0x80) != 0) {
                    byteList += 1
                    len = ((len and 0x7f) shl 8) or byteList.castToInt(0, 1)
                }
                byteList += 1
                return len
            }
        }

        override fun toString(): String {
            return string
        }
    }

    operator fun get(index: Int): String {
        return strings[index].string
    }

    data class StyleInfo(
        val data: List<Byte>,
        val spans: List<ResStringPoolSpan>
    ) {
        companion object {

            private const val END: Int = (0xFFFFFFFF).toInt()

            fun isEnd(byteList: ByteList): Boolean {
                return byteList.castToInt(0, 4) != END
            }
        }
    }
}

data class ResStringPoolRef(
    @unit_32_t
    val index: Int
) {
    private var _value: String? = null
    val value: String?
        get() {
            return _value
        }

    fun updateValue(stringPool: ResStringPool) {
        if (index < 0 || index >= stringPool.strings.size) {
            return
        }
        _value = stringPool[index]
    }
}

@ResStringPoolRefContainer
data class ResStringPoolSpan(
    @unit_size_t_group
    val ref: ResStringPoolRef,
    @unit_32_t
    val firstChar: Int,
    @unit_32_t
    val lastChar: Int
)

@Target(AnnotationTarget.CLASS)
annotation class ResStringPoolRefContainer

interface IResStringPoolRefContainer {
    fun updateWithResStringPool(resStringPool: ResStringPool)
}

val Class<*>.hasRefStringPoolRef: Boolean
    get() {
        return when {
            this == ResStringPoolRef::class.java -> true
            this.declaredAnnotations.any { it is ResStringPoolRefContainer } -> true
            else -> false
        }
    }

val Field.hasRefStringPoolRef: Boolean
    get() {
        return when {
            List::class.java.isAssignableFrom(type) -> {
                val genericType = genericType
                if (genericType !is ParameterizedType) {
                    false
                } else {
                    val valueType = genericType.actualTypeArguments[0]
                    if (valueType is Class<*>) {
                        valueType.hasRefStringPoolRef
                    } else {
                        false
                    }
                }
            }

            else -> type.hasRefStringPoolRef
        }
    }

fun Any.getAllResStringPoolRefs(): List<ResStringPoolRef>? {
    if (this is ResStringPoolRef) {
        return listOf(this)
    }
    if (this is List<*>) {
        return this.mapNotNull { it?.getAllResStringPoolRefs() }.flatten()
            .takeIf { it.isNotEmpty() }
    }
    return this::class.java.declaredFields
        .filter { it.hasRefStringPoolRef }
        .mapNotNull { it.also { it.isAccessible = true }.get(this) }
        .mapNotNull { value ->
            when (value) {
                is ResStringPoolRef -> listOf(value)
                else -> value.getAllResStringPoolRefs()
            }
        }
        .takeIf { it.isNotEmpty() }?.flatten()
}