/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

@file:Suppress("MemberVisibilityCanBePrivate")

package com.bytedance.ultimate.inflater.plugin.arsc

import java.util.*

/**
 * Created by ChengTao(chentao.joe@bytedance.com) on 2022/8/18.
 */

data class ResTableHeader(
    @unit_size_t_group
    val header: ResChunkHeader,
    @unit_32_t
    val packageCount: Int
)


class ResTable(data: ByteArray) {

    private lateinit var _header: ResTableHeader

    val header: ResTableHeader
        get() = _header

    val size: Int
        get() = header.header.size

    private lateinit var _stringPool: ResStringPool

    val stringPool: ResStringPool
        get() = _stringPool

    private val _resTablePackages = mutableListOf<ResTablePackage>()

    val resTablePackages: List<ResTablePackage>
        get() = Collections.unmodifiableList(_resTablePackages)


    init {
        addInternal(data)
        _resTablePackages.flatMap { p -> p.typeTypes.flatMap { it.value } }
            .map { it.entries }
            .forEach { entryMap ->
                entryMap.forEach { (_, entry) ->
                    when (entry) {
                        is ResValueResTableEntry -> entry.resValue.getAllResStringPoolRefs()
                        is ResTableMapEntry -> entry.values.getAllResStringPoolRefs()
                        else -> null
                    }?.forEach { ref ->
                        ref.updateValue(_stringPool)
                    }
                }
            }
        _resTablePackages
    }

    private fun addInternal(data: ByteArray) {
        val byteList = ByteList(data.toList())
        _header = cast(byteList)
        byteList += header.header.headerSize
        while (byteList.isValid) {
            val chunk: ResChunkHeader = cast(byteList)
            when (chunk.type) {
                ResXmlType.RES_STRING_POOL_TYPE -> _stringPool = cast(byteList)
                ResXmlType.RES_TABLE_PACKAGE_TYPE -> {
                    val resTablePackage = cast<ResTablePackage>(byteList)
                    _resTablePackages.add(resTablePackage)
                }

                else -> {}
            }
            byteList += chunk.size
        }
    }

    fun getLayoutId(layoutName: String): Int {
        return getResourceId("layout", layoutName)
    }

    fun getLayoutName(layoutId: Int): String {
        return getResourceName("layout", layoutId)
    }

    fun getResourceId(type: String, name: String): Int {
        _resTablePackages.forEach { p ->
            val resourceId = p.getResourceId(type, name)
            if (resourceId != null) {
                return resourceId
            }
        }
        throw IllegalStateException("Cannot get resource id with type $type and name $name")
    }

    fun getResourceName(type: String, id: Int): String {
        _resTablePackages.forEach { p ->
            val resourceName = p.getResourceName(type, id)
            if (resourceName != null) {
                return resourceName
            }
        }
        throw IllegalStateException("Cannot get resource name with type $type and id $id")
    }
}