/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

import java.lang.reflect.Field

/**
 * Created by ChengTao(chentao.joe@bytedance.com) on 2022/8/18.
 */
data class ResTablePackage(
    @unit_size_t_group
    val header: ResChunkHeader,
    @unit_32_t
    val id: Int,
    // Actual name of this package, \0-terminated
    @FieldCast(FieldNameCaster::class)
    val nameInfo: NameInfo,
    @unit_32_t
    val typeStrings: Int,
    @unit_32_t
    val lastPublicType: Int,
    @unit_32_t
    val keyStrings: Int,
    @unit_32_t
    val lastPublicKey: Int,
    @unit_32_t
    val typeIdOffSet: Int,
    @FieldCast(FieldTypeStringPoolCaster::class)
    val typeStringPool: ResStringPool,
    @FieldCast(FieldKeyStringPoolCaster::class)
    val keyStringPool: ResStringPool,
    @FieldCast(FieldTypeSpecsCaster::class)
    val typeSpecs: List<ResTableTypeSpec>,
    @FieldCast(FieldTypeTypesCaster::class)
    val typeTypes: Map<Int, List<ResTableType>>
) {

    val name: String = nameInfo.string

    private val resourceTypeToTypeId: Map<String, Int>
    private val typeIdToResourceId: Map<Int/*typeId*/, Map<String/*name*/, ResourceId>>
    private val typeIdToResourceName: Map<Int/*typeId*/, Map<Int/*resourceId*/, String/*resourceName*/>>

    init {
        if (id == 0) {
            throw IllegalStateException("invalid id")
        }
        // update all type entries' key
        typeTypes.map { it.value }.flatten()
            .map { it.entries.values }.flatten()
            .map { it.entry.key }.forEach { ref ->
                ref.updateValue(keyStringPool)
            }
        // store all resource type
        resourceTypeToTypeId = typeStringPool.strings.map { it.string }
            .mapIndexed { index, type -> type to index + 1 }.toMap()
        // create resource's id
        val packageId = id
        typeIdToResourceId = typeTypes.map { (typeId, typeList) ->
            val nameToResourceId = mutableMapOf<String, ResourceId>()
            typeList.map { it.entries.entries }.flatten().forEach { (entryId, entry) ->
                val resourceName = entry.entry.key.value
                if (resourceName != null) {
                    // little-endian
                    val entryIdByteList = listOf(
                        /*entryId*/
                        entryId.toByteList(2),
                        /*typeId*/
                        typeId.toByteList(1),
                        /*packageId*/
                        packageId.toByteList(1)
                    ).flatten()
                    nameToResourceId[resourceName] =
                        ResourceId(entryIdByteList.toInt(), entryIdByteList.toHex())
                }
            }
            typeId to nameToResourceId
        }.toMap()
        typeIdToResourceName = typeIdToResourceId.map { entry ->
            val typeId = entry.key
            val value = entry.value
            typeId to value.map { it.value.value to it.key }.toMap()
        }.toMap()
    }

    fun getResourceId(type: String, name: String): Int? {
        val typeId = resourceTypeToTypeId[type] ?: return null
        return typeIdToResourceId[typeId]?.get(name)?.value
    }

    fun getResourceName(type: String, id: Int): String? {
        val typeId = resourceTypeToTypeId[type] ?: return null
        return typeIdToResourceName[typeId]?.get(id)
    }

    data class NameInfo(
        val data: List<Byte>,
        val string: String
    )

    data class ResourceId(
        val value: Int,
        val hex: String
    )

    class FieldNameCaster : DefaultFieldCaster<NameInfo>() {

        override fun getFieldSize(field: Field, value: NameInfo) = FIELD_SIZE

        override fun cast(field: Field, byteList: ByteList): NameInfo {
            val data = byteList.subList(0, FIELD_SIZE)
            val string = data.chunked(2)
                .map { it.toInt() }
                .dropLastWhile { it == 0 }
                .map { it.toChar() }
                .joinToString("") { it.toString() }
            return NameInfo(data, string)
        }

        companion object {
            private const val FIELD_SIZE = 2 * 128
        }
    }

    abstract class AbsStringPoolCaster : DefaultFieldCaster<ResStringPool>() {

        private val headerIndex by lazy { getFieldIndex("header") }

        abstract val fromOffsetFiledName: String

        override fun getFieldSize(field: Field, value: ResStringPool): Int {
            return value.header.header.size
        }

        final override fun cast(field: Field, byteList: ByteList): ResStringPool {
            val fromOffset = getFieldValue<Int>(fromOffsetFiledName)
            byteList += (headerIndex - byteList.currentIndex + fromOffset)
            return cast(byteList)
        }
    }

    class FieldTypeStringPoolCaster : AbsStringPoolCaster() {
        override val fromOffsetFiledName: String
            get() = "typeStrings"
    }

    class FieldKeyStringPoolCaster : AbsStringPoolCaster() {
        override val fromOffsetFiledName: String
            get() = "keyStrings"
    }

    class FieldTypeSpecsCaster : DefaultFieldCaster<List<ResTableTypeSpec>>() {

        private val header: ResChunkHeader by lazy { getFieldValue<ResChunkHeader>("header") }

        private val headerIndex by lazy { getFieldIndex("header") }
        override fun cast(field: Field, byteList: ByteList): List<ResTableTypeSpec> {
            byteList += (headerIndex - byteList.currentIndex + header.headerSize)
            val list = mutableListOf<ResTableTypeSpec>()
            while (byteList.isValid) {
                val chunk: ResChunkHeader = cast(byteList)
                if (chunk.type == ResXmlType.RES_TABLE_TYPE_SPEC_TYPE) {
                    list.add(cast(byteList))
                }
                byteList += chunk.size
            }
            return list
        }
    }

    class FieldTypeTypesCaster : DefaultFieldCaster<Map<Int, List<ResTableType>>>() {

        private val header: ResChunkHeader by lazy { getFieldValue<ResChunkHeader>("header") }

        private val headerIndex by lazy { getFieldIndex("header") }
        override fun cast(field: Field, byteList: ByteList): Map<Int, List<ResTableType>> {
            byteList += (headerIndex - byteList.currentIndex + header.headerSize)
            val map = mutableMapOf<Int, MutableList<ResTableType>>()
            while (byteList.isValid) {
                val chunk: ResChunkHeader = cast(byteList)
                if (chunk.type == ResXmlType.RES_TABLE_TYPE_TYPE) {
                    val type = cast<ResTableType>(byteList)
                    map.getOrPut(type.id) { mutableListOf() }.add(type)
                }
                byteList += chunk.size
            }
            return map
        }
    }
}