/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

import java.lang.reflect.Field
import java.util.*

/**
 * Created by chentao.joe on 2022/8/20
 * @author chentao.joe@bytedance.com
 */
data class ResTableType(
    @unit_size_t_group
    val header: ResChunkHeader,
    @unit_8_t
    val id: Int,
    @unit_8_t
    val flags: Int,
    @unit_16_t
    val reserved: Int,
    @unit_32_t
    val entryCount: Int,
    @unit_32_t
    val entriesStart: Int,
    @unit_size_t_group
    @FieldCast(FieldConfigCaster::class)
    val config: ResTableConfig,
    @FieldCast(FieldEntriesCaster::class)
    val entries: Map<Int, IResTableEntry>
) {

    val isSpareEntry: Boolean
        get() = flags == FLAG_SPARE

    class FieldConfigCaster : DefaultFieldCaster<ResTableConfig>() {
        override fun getFieldSize(field: Field, value: ResTableConfig): Int {
            return value.size
        }

        override fun cast(field: Field, byteList: ByteList): ResTableConfig {
            return cast(byteList)
        }
    }

    class FieldEntriesCaster : DefaultFieldCaster<Map<Int, IResTableEntry>>() {

        private val header: ResChunkHeader by lazy { getFieldValue<ResChunkHeader>("header") }
        private val flags: Int by lazy { getFieldValue<Int>("flags") }
        private val entryCount: Int by lazy { getFieldValue<Int>("entryCount") }
        private val entriesStart: Int by lazy { getFieldValue<Int>("entriesStart") }

        private val headerIndex by lazy { getFieldIndex("header") }

        private val configIndex by lazy { getFieldIndex("config") }
        private val configSize by lazy { getFieldValue<ResTableConfig>("config").size }

        override fun getFieldSize(field: Field, value: Map<Int, IResTableEntry>): Int {
            return (headerIndex + header.size) - (configIndex + configSize)
        }

        override fun cast(field: Field, byteList: ByteList): Map<Int, IResTableEntry> {
            if (flags == FLAG_SPARE || entryCount <= 0) {
                return emptyMap()
            }
            val entryOffsetByteList = byteList.subList(
                configIndex, configSize, headerIndex + entriesStart - configIndex
            ).toByteList()
            if (entryOffsetByteList.size / 4 != entryCount) {
                return emptyMap()
            }
            val entryByteList = byteList.subList(
                headerIndex, entriesStart, header.size
            ).toByteList()
            val entries = TreeMap<Int, IResTableEntry>()
            for (i in 0 until entryCount) {
                val entryOffset = entryOffsetByteList.castToIntAndPlusAssign(4)
                if (entryOffset == IResTableEntry.NO_ENTRY) {
                    continue
                }
                entryByteList += entryOffset
                entries[i] = IResTableEntry.parse(entryByteList)
                entryByteList -= entryOffset
            }
            return entries
        }
    }

    companion object {
        private const val FLAG_SPARE = 0x01
    }
}

data class ResTableConfig(
    @unit_32_t
    val size: Int,
    @unit_size_t_group
    val imsi: Imsi,
    @unit_size_t_group
    val locate: Locale,
    @unit_size_t_group
    val screenType: ScreenType,
    @unit_size_t_group
    val input: Input,
    @unit_size_t_group
    val screenSize: ScreenSize,
    @unit_size_t_group
    val version: Version,
    @unit_size_t_group
    val screenConfig: ScreenConfig,
    @unit_size_t_group
    val screenSizeDp: ScreenSizeDp,
    @FieldCast(FieldLocalScriptCaster::class)
    val localScript: StringValue,
    @FieldCast(FieldLocalVariantCaster::class)
    val localVariant: StringValue,
    @unit_size_t_group
    val screenConfig2: ScreenConfig2
) {
    data class Imsi(
        @unit_16_t
        val mcc: Int,
        @unit_16_t
        val mnc: Int
    )

    data class StringValue(
        val data: List<Byte>,
        val value: String
    )

    abstract class StringFieldCaster : DefaultFieldCaster<StringValue>() {

        abstract val stringSize: Int

        final override fun getFieldSize(field: Field, value: StringValue) = stringSize

        final override fun cast(field: Field, byteList: ByteList): StringValue {
            val data = byteList.subList(0, stringSize)
            return StringValue(data, createString(data.toByteArray()))
        }

        open fun createString(data: ByteArray) = String(data)
    }


    data class Locale(
        @unit_16_t
        @FieldCast(FieldLanguageCaster::class)
        val language: StringValue,
        @unit_16_t
        @FieldCast(FieldCountryCaster::class)
        val country: StringValue
    ) {
        class FieldLanguageCaster : StringFieldCaster() {
            override val stringSize: Int
                get() = 2

            override fun createString(data: ByteArray): String {
                if (data[0] == 0.toByte() && data[1] == 0.toByte()) {
                    return "any"
                }
                return super.createString(data)
            }
        }

        class FieldCountryCaster : StringFieldCaster() {
            override val stringSize: Int
                get() = 2
        }
    }

    data class ScreenType(
        @unit_8_t
        val orientation: Int,
        @unit_8_t
        val touchScreen: Int,
        @unit_16_t
        val density: Int
    )

    data class Input(
        @unit_8_t
        val keyboard: Int,
        @unit_8_t
        val navigation: Int,
        @unit_8_t
        val inputFlags: Int,
        @unit_8_t
        val inputPad: Int
    )

    data class ScreenSize(
        @unit_16_t
        val screenWidth: Int,
        @unit_16_t
        val screenHeight: Int
    )

    data class Version(
        @unit_16_t
        val sdkVersion: Int,
        @unit_16_t
        val minorVersion: Int
    )

    data class ScreenConfig(
        @unit_8_t
        val screenLayout: Int,
        @unit_8_t
        val uiMode: Int,
        @unit_16_t
        val smallestScreenWidthDp: Int
    )

    data class ScreenSizeDp(
        @unit_16_t
        val screenWidthDp: Int,
        @unit_16_t
        val screenHeightDp: Int
    )

    class FieldLocalScriptCaster : StringFieldCaster() {
        override val stringSize: Int
            get() = 4
    }

    class FieldLocalVariantCaster : StringFieldCaster() {
        override val stringSize: Int
            get() = 8
    }

    data class ScreenConfig2(
        @unit_8_t
        val screenLayout2: Int,
        @unit_8_t
        val uiMode: Int,
        @unit_16_t
        val screenConfigPad2: Int
    )
}


@ResStringPoolRefContainer
data class ResTableEntry(
    @unit_16_t
    val size: Int,
    @unit_16_t
    val flags: Int,
    @unit_size_t_group
    val key: ResStringPoolRef
) {
    companion object {
        //=============== Flags Start ===============//

        // If set, this is a complex entry, holding a set of name/value
        // mappings.  It is followed by an array of ResTable_map structures.
        const val FLAG_COMPLEX = 0x0001

        // If set, this resource has been declared public, so libraries
        // are allowed to reference it.
        const val FLAG_PUBLIC = 0x0002

        // If set, this is a weak resource and may be overriden by strong
        // resources of the same name/type. This is only useful during
        // linking with other resource tables.
        const val FLAG_WEAK = 0x0004

        //=============== Flags End ===============//
    }
}

interface IResTableEntry {
    val entry: ResTableEntry

    companion object {

        const val NO_ENTRY = (0xFFFFFFFF).toInt()

        fun parse(byteList: ByteList): IResTableEntry {
            val entry = cast<ResTableEntry>(byteList)
            return if (entry.flags.and(ResTableEntry.FLAG_COMPLEX) != 0) {
                cast<ResTableMapEntry>(byteList)
            } else {
                cast<ResValueResTableEntry>(byteList)
            }
        }
    }
}

@ResStringPoolRefContainer
data class ResValueResTableEntry(
    @unit_size_t_group
    override val entry: ResTableEntry,
    @unit_size_t_group
    val resValue: ResValue
) : IResTableEntry


data class ResTableMapEntry(
    @unit_size_t_group
    override val entry: ResTableEntry,
    @unit_size_t_group
    val parent: ResStringPoolRef,
    @unit_32_t
    val count: Int,
    @FieldCast(FieldValuesCaster::class)
    val values: List<ResTableMap>
) : IResTableEntry {
    class FieldValuesCaster : DefaultFieldCaster<List<ResTableMap>>() {

        private val count: Int by lazy { getFieldValue<Int>("count") }

        private val resTableMapSize by lazy { sizeOf(ResTableMap::class) }

        override fun cast(field: Field, byteList: ByteList): List<ResTableMap> {
            return byteList.subList(0, count * resTableMapSize)
                .chunked(resTableMapSize)
                .map { cast<ResTableMap>(ByteList(it)) }
        }
    }
}

@ResStringPoolRefContainer
data class ResTableMap(
    @unit_size_t_group
    val name: ResStringPoolRef,
    @unit_size_t_group
    val value: ResValue
)
