/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

import java.lang.reflect.Field


/**
 * Created by chentao.joe on 2022/7/18
 * @author chentao.joe@bytedance.com
 */
enum class ResValueType(val value: Int) {

    // The 'data' is either 0 or 1, specifying this resource is either
    // undefined or empty, respectively.
    TYPE_NULL(0x00),

    // The 'data' holds a ResTable_ref), a reference to another resource
    // table entry.
    TYPE_REFERENCE(0x01),

    // The 'data' holds an attribute resource identifier.
    TYPE_ATTRIBUTE(0x02),

    // The 'data' holds an index into the containing resource table's
    // global value string pool.
    TYPE_STRING(0x03),

    // The 'data' holds a single-precision floating point number.
    TYPE_FLOAT(0x04),

    // The 'data' holds a complex number encoding a dimension value),
    // such as "100in".
    TYPE_DIMENSION(0x05),

    // The 'data' holds a complex number encoding a fraction of a
    // container.
    TYPE_FRACTION(0x06),

    // The 'data' holds a dynamic ResTable_ref), which needs to be
    // resolved before it can be used like a TYPE_REFERENCE.
    TYPE_DYNAMIC_REFERENCE(0x07),

    // The 'data' holds an attribute resource identifier), which needs to be resolved
    // before it can be used like a TYPE_ATTRIBUTE.
    TYPE_DYNAMIC_ATTRIBUTE(0x08),

    // Beginning of integer flavors...
    TYPE_FIRST_INT(0x10),

    // The 'data' is a raw integer value of the form n..n.
    TYPE_INT_DEC(0x10),

    // The 'data' is a raw integer value of the form 0xn..n.
    TYPE_INT_HEX(0x11),

    // The 'data' is either 0 or 1), for input "false" or "true" respectively.
    TYPE_INT_BOOLEAN(0x12),

    // Beginning of color integer flavors...
    TYPE_FIRST_COLOR_INT(0x1c),

    // The 'data' is a raw integer value of the form #aarrggbb.
    TYPE_INT_COLOR_ARGB8(0x1c),

    // The 'data' is a raw integer value of the form #rrggbb.
    TYPE_INT_COLOR_RGB8(0x1d),

    // The 'data' is a raw integer value of the form #argb.
    TYPE_INT_COLOR_ARGB4(0x1e),

    // The 'data' is a raw integer value of the form #rgb.
    TYPE_INT_COLOR_RGB4(0x1f),

    // ...end of integer flavors.
    TYPE_LAST_COLOR_INT(0x1f),

    // ...end of integer flavors.
    TYPE_LAST_INT(0x1f)
}

internal fun Int.toResValueType(): ResValueType {
    return ResValueType.values().find { it.value == this }
        ?: throw IllegalStateException("Invalid ResValueType : $this")
}

enum class Dimension(val value: Int) {
    // Where the unit type information is.  This gives us 16 possible
    // types, as defined below.
    COMPLEX_UNIT_SHIFT(0),
    COMPLEX_UNIT_MASK(0xf),

    // TYPE_DIMENSION: Value is raw pixels.
    COMPLEX_UNIT_PX(0),

    // TYPE_DIMENSION: Value is Device Independent Pixels.
    COMPLEX_UNIT_DIP(1),

    // TYPE_DIMENSION: Value is a Scaled device independent Pixels.
    COMPLEX_UNIT_SP(2),

    // TYPE_DIMENSION: Value is in points.
    COMPLEX_UNIT_PT(3),

    // TYPE_DIMENSION: Value is in inches.
    COMPLEX_UNIT_IN(4),

    // TYPE_DIMENSION: Value is in millimeters.
    COMPLEX_UNIT_MM(5),

    // TYPE_FRACTION: A basic fraction of the overall size.
    COMPLEX_UNIT_FRACTION(0),

    // TYPE_FRACTION: A fraction of the parent size.
    COMPLEX_UNIT_FRACTION_PARENT(1),

    // Where the radix information is), telling where the decimal place
    // appears in the mantissa.  This give us 4 possible fixed point
    // representations as defined below.
    COMPLEX_RADIX_SHIFT(4),
    COMPLEX_RADIX_MASK(0x3),

    // The mantissa is an integral number -- i.e.), 0xnnnnnn.0
    COMPLEX_RADIX_23p0(0),

    // The mantissa magnitude is 16 bits -- i.e), 0xnnnn.nn
    COMPLEX_RADIX_16p7(1),

    // The mantissa magnitude is 8 bits -- i.e), 0xnn.nnnn
    COMPLEX_RADIX_8p15(2),

    // The mantissa magnitude is 0 bits -- i.e), 0x0.nnnnnn
    COMPLEX_RADIX_0p23(3),

    // Where the actual value is.  This gives us 23 bits of
    // precision.  The top bit is the sign.
    COMPLEX_MANTISSA_SHIFT(8),
    COMPLEX_MANTISSA_MASK(0xffffff)
}

private val MANTISSA_MULT = 1.0f / (1 shl Dimension.COMPLEX_MANTISSA_SHIFT.value)
private val RADIX_MULTS = floatArrayOf(
    1.0f * MANTISSA_MULT,
    1.0f / (1 shl 7) * MANTISSA_MULT,
    1.0f / (1 shl 15) * MANTISSA_MULT,
    1.0f / (1 shl 23) * MANTISSA_MULT
)

internal fun Int.toDimension(): Dimension {
    val unit = (this shr Dimension.COMPLEX_UNIT_SHIFT.value) and Dimension.COMPLEX_UNIT_MASK.value
    return Dimension.values().find { it.value == unit }
        ?: throw IllegalStateException("Unknown dimension : $this")
}

private fun Int.complexToFloat(): Float {
    return (this and (Dimension.COMPLEX_MANTISSA_MASK.value shl Dimension.COMPLEX_MANTISSA_SHIFT.value)) *
            (RADIX_MULTS[(this shr Dimension.COMPLEX_RADIX_SHIFT.value) and Dimension.COMPLEX_RADIX_MASK.value])
}

@ResStringPoolRefContainer
interface ResValueData<V> {
    val data: List<Byte>
    val value: V
}

data class IntResValueData(
    override val data: List<Byte>,
    override val value: Int
) : ResValueData<Int>

data class FloatResValueData(
    override val data: List<Byte>,
    override val value: Float
) : ResValueData<Float>

data class FractionResValueData(
    override val data: List<Byte>,
    val fraction: Float
) : ResValueData<Float> {
    override val value: Float
        get() = fraction
}

data class ReferenceResValueData(
    override val data: List<Byte>,
    val referenceIdHex: String
) : ResValueData<String> {
    override val value: String
        get() = referenceIdHex
}

@ResStringPoolRefContainer
data class StringPoolResValueData(
    override val data: List<Byte>,
    val index: ResStringPoolRef
) : ResValueData<String> {
    override val value: String
        get() = index.value!!
}

data class ColorResValueData(
    override val data: List<Byte>,
    val colorHex: String
) : ResValueData<String> {
    override val value: String
        get() = colorHex
}

data class DimensionResValueData(
    override val data: List<Byte>,
    val dimension: Dimension,
    override val value: Float
) : ResValueData<Float>

data class HexResValueData(
    override val data: List<Byte>,
    val hex: String
) : ResValueData<String> {
    override val value: String
        get() = hex
}

@ResStringPoolRefContainer
data class ResValue(
    @unit_16_t
    val size: Int,
    @unit_8_t
    val res0: Int,
    @unit_8_t
    @FieldCast(DataTypeFieldCaster::class)
    val dataType: ResValueType,
    @FieldCast(DataFieldCaster::class)
    @unit_32_t
    val data: ResValueData<*>
) {

    class DataTypeFieldCaster : IntFieldTransformCaster<ResValueType>() {
        override fun castTransform(value: Int) = value.toResValueType()
        override fun unCastTransform(value: ResValueType) = value.value
    }


    class DataFieldCaster : AbsFieldCaster<ResValueData<*>>() {

        private val dataType: ResValueType by lazy { getFieldValue<ResValueType>("dataType") }

        override fun getFieldSize(field: Field, value: ResValueData<*>) = 4

        override fun cast(field: Field, byteList: ByteList): ResValueData<*> {
            return parseData(byteList.subList(0, 4), dataType)
        }

        override fun unCast(
            field: Field,
            value: ResValueData<*>,
            interceptor: UnCastInterceptor
        ): List<Byte> {
            if (value !is StringPoolResValueData) {
                return value.data
            }
            return interceptor.intercept(
                StringPoolResValueData::class.java.getDeclaredField("index"),
                value.index
            ) ?: throw IllegalStateException("Cannot unCast StringPoolResValueData : $value")
        }
    }

    companion object {
        fun parseData(data: List<Byte>, dataType: ResValueType): ResValueData<*> {
            if (data.size != 4) {
                throw IllegalArgumentException("size of data must be 4.")
            }
            return when (dataType) {
                // RawResValueData
                ResValueType.TYPE_NULL,
                ResValueType.TYPE_INT_DEC,
                ResValueType.TYPE_INT_BOOLEAN,
                ResValueType.TYPE_FIRST_INT,
                ResValueType.TYPE_LAST_INT -> {
                    IntResValueData(data, data.toInt())
                }

                ResValueType.TYPE_FLOAT -> {
                    val value = data.toInt()
                    FloatResValueData(data, Float.fromBits(value))
                }

                ResValueType.TYPE_FRACTION -> {
                    val complex = data.toInt()
                    FractionResValueData(data, complex.complexToFloat())
                }
                // ReferenceResValueData
                ResValueType.TYPE_REFERENCE,
                ResValueType.TYPE_DYNAMIC_REFERENCE,
                ResValueType.TYPE_ATTRIBUTE,
                ResValueType.TYPE_DYNAMIC_ATTRIBUTE -> {
                    val referenceIdHex = data.toHex()
                    ReferenceResValueData(data, referenceIdHex)
                }
                // StringPoolResValueData
                ResValueType.TYPE_STRING -> {
                    val index = data.toInt()
                    StringPoolResValueData(data, ResStringPoolRef(index))
                }

                ResValueType.TYPE_DIMENSION -> {
                    val complex = data.toInt()
                    val dimension = complex.toDimension()
                    val value = complex.complexToFloat()
                    return DimensionResValueData(data, dimension, value)
                }

                ResValueType.TYPE_INT_HEX -> {
                    val hex = data.toHex()
                    HexResValueData(data, hex)
                }

                ResValueType.TYPE_FIRST_COLOR_INT,
                ResValueType.TYPE_INT_COLOR_ARGB8,
                ResValueType.TYPE_INT_COLOR_RGB8,
                ResValueType.TYPE_INT_COLOR_ARGB4,
                ResValueType.TYPE_INT_COLOR_RGB4,
                ResValueType.TYPE_LAST_COLOR_INT -> {
                    val colorHex = data.toHex()
                    ColorResValueData(data, colorHex)
                }

                else -> {
                    throw IllegalStateException("Unknown Type : $dataType")
                }
            }
        }
    }
}
