/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

/**
 * Created by chentao.joe on 2022/7/20
 * @author chentao.joe@bytedance.com
 */

enum class ResXmlType(val value: Int) {
    RES_NULL_TYPE(0x0000),
    RES_STRING_POOL_TYPE(0x0001),
    RES_TABLE_TYPE(0x0002),
    RES_XML_TYPE(0x0003),

    // Chunk types in RES_XML_TYPE
    RES_XML_FIRST_CHUNK_TYPE(0x0100),
    RES_XML_START_NAMESPACE_TYPE(0x0100),
    RES_XML_END_NAMESPACE_TYPE(0x0101),
    RES_XML_START_ELEMENT_TYPE(0x0102),
    RES_XML_END_ELEMENT_TYPE(0x0103),
    RES_XML_CDATA_TYPE(0x0104),
    RES_XML_LAST_CHUNK_TYPE(0x017f),

    // This contains a uint32_t array mapping strings in the string
    // pool back to resource identifiers.  It is optional.
    RES_XML_RESOURCE_MAP_TYPE(0x0180),

    // Chunk types in RES_TABLE_TYPE
    RES_TABLE_PACKAGE_TYPE(0x0200),
    RES_TABLE_TYPE_TYPE(0x0201),
    RES_TABLE_TYPE_SPEC_TYPE(0x0202),
    RES_TABLE_LIBRARY_TYPE(0x0203),
    RES_TABLE_OVERLAYABLE_TYPE(0x0204),
    RES_TABLE_OVERLAYABLE_POLICY_TYPE(0x0205),
    RES_TABLE_STAGED_ALIAS_TYPE(0x0206);
}

val ResXmlType.isNamespace: Boolean
    get() {
        return when (this) {
            ResXmlType.RES_XML_FIRST_CHUNK_TYPE,
            ResXmlType.RES_XML_START_NAMESPACE_TYPE,
            ResXmlType.RES_XML_END_NAMESPACE_TYPE -> true

            else -> false
        }
    }

val ResXmlType.isElement: Boolean
    get() {
        return when (this) {
            ResXmlType.RES_XML_START_ELEMENT_TYPE,
            ResXmlType.RES_XML_END_ELEMENT_TYPE -> true

            else -> false
        }
    }

internal fun Int.toResXmlType(): ResXmlType {
    val type = ResXmlType.values().find { it.value == this }
        ?: throw IllegalStateException("Unknown ResXmlType : $this")
    // FIRST_CHUNK_TYPE is same as START_NAMESPACE_TYPE
    if (type == ResXmlType.RES_XML_FIRST_CHUNK_TYPE) {
        return ResXmlType.RES_XML_START_NAMESPACE_TYPE
    }
    return type
}

data class ResChunkHeader(
    @unit_16_t
    @FieldCast(TypeCaster::class)
    val type: ResXmlType,
    @unit_16_t
    val headerSize: Int,
    @unit_32_t
    val size: Int
) {
    class TypeCaster : IntFieldTransformCaster<ResXmlType>() {
        override fun castTransform(value: Int) = value.toResXmlType()
        override fun unCastTransform(value: ResXmlType) = value.value
    }
}

val ResChunkHeader.isNamespace: Boolean
    get() {
        return type.isNamespace
    }

val ResChunkHeader.isElement: Boolean
    get() {
        return type.isElement
    }

val ResChunkHeader.isTreeNode: Boolean
    get() {
        return isNamespace || isElement
    }

fun <T> ByteList.chunkMapNotNull(action: (header: ResChunkHeader, chunk: ByteList) -> T?): List<T> {
    val resultList = mutableListOf<T>()
    val byteList = this
    while (byteList.isValid) {
        val header = cast<ResChunkHeader>(byteList)
        val chunk = ByteList(byteList.subList(0, header.size))
        action(header, chunk)?.let { mapped ->
            resultList.add(mapped)
        }
        byteList += header.size
    }
    return resultList
}