/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

import java.lang.reflect.Field


/**
 * Created by chentao.joe on 2022/7/20
 * @author chentao.joe@bytedance.com
 */
data class ResXmlResourceMap(
    @unit_size_t_group
    val header: ResChunkHeader,
    @FieldCast(ResIdsFieldCaster::class)
    val resIds: List<ResId>
) {

    data class ResId(
        val id: String,
        val data: List<Byte>
    )

    class ResIdsFieldCaster : AbsFieldCaster<List<ResId>>() {

        private val header: ResChunkHeader by lazy { getFieldValue<ResChunkHeader>("header") }
        private val headerIndex by lazy { getFieldIndex("header") }

        override fun getFieldSize(field: Field, value: List<ResId>): Int {
            return value.sumOf { it.data.size }
        }

        override fun cast(field: Field, byteList: ByteList): List<ResId> {
            return byteList.subList(headerIndex, header.headerSize, header.size)
                .also { if (it.size % 4 != 0) throw IllegalStateException("Invalid resIds") }
                .chunked(4)
                .map { ResId(it.toHex(), it) }
        }

        override fun unCast(
            field: Field,
            value: List<ResId>,
            interceptor: UnCastInterceptor
        ): List<Byte> {
            return interceptor.intercept(field, value)
                ?: throw IllegalStateException("Cannot unCast resIds.")
        }
    }
}