/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

import com.bytedance.ultimate.inflater.plugin.arsc.single.SingleNodeResXmlTree
import java.util.*

/**
 * Created by chentao.joe on 2022/7/19
 * @author chentao.joe@bytedance.com
 */
data class ResXmlTreeHeader(
    @unit_size_t_group
    val header: ResChunkHeader
)

@Suppress("MemberVisibilityCanBePrivate")
class ResXmlTree {

    private lateinit var _dataList: List<Byte>

    val dataList: List<Byte>
        get() = Collections.unmodifiableList(_dataList)

    private lateinit var _header: ResXmlTreeHeader
    val header: ResXmlTreeHeader
        get() = _header

    private var _size = 0
    val size: Int
        get() = _size

    private lateinit var _stringPool: ResStringPool

    val stringPool: ResStringPool
        get() = _stringPool

    private lateinit var _resourceMap: ResXmlResourceMap

    val resourceMap: ResXmlResourceMap
        get() = _resourceMap

    // include may have no namespace
    private val _namespaces = mutableListOf<ResXmlTreeNodeNamespace>()

    val namespaces: List<ResXmlTreeNodeNamespace>
        get() = Collections.unmodifiableList(_namespaces)


    private lateinit var _rootNode: ResXmlTreeNodeElement

    val rootNode: ResXmlTreeNodeElement
        get() = _rootNode

    fun setTo(data: ByteArray) {
        setTo(data.toList())
    }

    fun setTo(dataList: List<Byte>) {
        _dataList = dataList
        val byteList = ByteList(dataList)
        _header = cast(byteList)
        _size = header.header.size
        if (header.header.headerSize > size || size > byteList.size) {
            throw IllegalStateException(
                "header size ${header.header.headerSize} or "
                        + "total size $size is large than data size : ${byteList.size}"
            )
        }
        byteList += header.header.headerSize
        byteList.chunkMapNotNull { header, chunk ->
            when (header.type) {
                // StringPool
                ResXmlType.RES_STRING_POOL_TYPE -> {
                    _stringPool = cast(chunk)
                }

                // ResourceMap
                ResXmlType.RES_XML_RESOURCE_MAP_TYPE -> {
                    _resourceMap = cast(chunk)
                }
                // TreeNode
                ResXmlType.RES_XML_FIRST_CHUNK_TYPE,
                ResXmlType.RES_XML_START_NAMESPACE_TYPE,
                ResXmlType.RES_XML_END_NAMESPACE_TYPE,
                ResXmlType.RES_XML_START_ELEMENT_TYPE,
                ResXmlType.RES_XML_END_ELEMENT_TYPE -> {
                    castAsTreeNode(chunk)
                }

                else -> null
            }
        }.also { chunkList ->
            parseResXmlTreedNode(chunkList.filterIsInstance<IResXmlTreeNode>())
        }
        // update with ResStringPool
        namespaces.forEach { it.updateWithResStringPool(stringPool) }
        rootNode.updateWithResStringPool(stringPool)
    }

    private fun parseResXmlTreedNode(treeNodeList: List<IResXmlTreeNode>) {
        if (treeNodeList.size % 2 != 0) {
            throw IllegalStateException("Invalid size of ResXmlTreeNode.")
        }
        val namespaceList = mutableListOf<ResXmlTreeNodeNamespace>()
        val namespaceStack: Stack<ResXmlTreeNodeNamespace> = Stack()
        val elementStack: Stack<ResXmlTreeNodeElement> = Stack()
        treeNodeList.forEach { node ->
            when (node) {
                is ResXmlTreeStartNamespace -> {
                    val namespace = ResXmlTreeNodeNamespace()
                    namespace.setStart(node)
                    namespaceList.add(namespace)
                    namespaceStack.push(namespace)
                }

                is ResXmlTreeEndNamespace -> {
                    val namespace = namespaceStack.pop()
                    namespace.setEnd(node)
                }

                is ResXmlTreeStartElement -> {
                    val element = ResXmlTreeNodeElement()
                    element.setStart(node)
                    elementStack.push(element)
                }

                is ResXmlTreeEndElement -> {
                    val element = elementStack.pop()
                    element.setEnd(node)
                    // parent
                    val parent = if (elementStack.empty()) null else elementStack.peek()
                    if (parent != null) {
                        element.setParent(parent)
                    } else {
                        if (this::_rootNode.isInitialized) {
                            throw IllegalStateException("Can not set rootNode twice.")
                        }
                        _rootNode = element
                    }
                }

                else -> {
                    throw IllegalStateException("Invalid node : $node")
                }
            }
        }
        _namespaces.also { it.clear() }.addAll(namespaceList)
    }

    fun asSingleNodeResXmlTree(layoutFullName: String): SingleNodeResXmlTree {
        return SingleNodeResXmlTree.parse(this, layoutFullName)
    }
}

fun ByteArray.toResXmlTree(): ResXmlTree {
    return ResXmlTree().also { it.setTo(this) }
}

fun List<Byte>.toResXmlTree(): ResXmlTree {
    return ResXmlTree().also { it.setTo(this) }
}