/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

import com.android.build.gradle.internal.utils.toImmutableList
import com.android.build.gradle.internal.utils.toImmutableMap
import java.lang.reflect.Field

/**
 * Created by chentao.joe on 2022/7/20
 * @author chentao.joe@bytedance.com
 */

// region : ResXmlTreeNode
interface IResXmlTreeNode : IResStringPoolRefContainer {
    override fun updateWithResStringPool(resStringPool: ResStringPool) {
        this.getAllResStringPoolRefs()?.forEach { ref ->
            ref.updateValue(resStringPool)
        }
    }
}

@ResStringPoolRefContainer
data class ResXmlTreeNode(
    @unit_size_t_group
    val header: ResChunkHeader,
    @unit_32_t
    val lineNumber: Int,
    @unit_size_t_group
    val comment: ResStringPoolRef
)

// endregion

// region : ResXmlTreeStartElement
@ResStringPoolRefContainer
data class ResXmlTreeAttrExt(
    @unit_size_t_group
    val ns: ResStringPoolRef,
    @unit_size_t_group
    val name: ResStringPoolRef,
    @unit_16_t
    val attributeStart: Int,
    @unit_16_t
    val attributeSize: Int,
    @unit_16_t
    val attributeCount: Int,
    @unit_16_t
    val idIndex: Int,
    @unit_16_t
    val classIndex: Int,
    @unit_16_t
    val styleIndex: Int
)

@ResStringPoolRefContainer
data class ResXmlTreeAttribute(
    @unit_size_t_group
    val ns: ResStringPoolRef,
    @unit_size_t_group
    val name: ResStringPoolRef,
    @unit_32_t
    val rawValue: Int,
    @unit_size_t_group
    val resValue: ResValue
)

data class ResXmlTreeStartElement(
    @unit_size_t_group
    val node: ResXmlTreeNode,
    @unit_size_t_group
    val attrExt: ResXmlTreeAttrExt,
    @FieldCast(AttributeListFieldCaster::class)
    val attributeList: List<ResXmlTreeAttribute>
) : IResXmlTreeNode {

    val name by lazy { attrExt.name.value!! }

    val attributes by lazy {
        attributeList.associate { it.name.value!! to it.resValue.data.value!! }.toImmutableMap()
    }

    class AttributeListFieldCaster : AbsFieldCaster<List<ResXmlTreeAttribute>>() {

        private val attrExt: ResXmlTreeAttrExt by lazy { getFieldValue<ResXmlTreeAttrExt>("attrExt") }
        private val attributeDataSize by lazy { attrExt.attributeSize * attrExt.attributeCount }

        override fun getFieldSize(field: Field, value: List<ResXmlTreeAttribute>): Int {
            return attributeDataSize
        }

        override fun cast(field: Field, byteList: ByteList): List<ResXmlTreeAttribute> {
            return byteList.subList(0, attributeDataSize).chunked(attrExt.attributeSize)
                .map { cast<ResXmlTreeAttribute>(ByteList(it)) }
        }

        override fun unCast(
            field: Field,
            value: List<ResXmlTreeAttribute>,
            interceptor: UnCastInterceptor
        ): List<Byte> {
            return value.map { it.unCast(interceptor) }.flatten()
        }
    }
}
// endregion


// region ResXmlEndElement
@ResStringPoolRefContainer
data class ResXmlTreeEndElementExt(
    @unit_size_t_group
    val ns: ResStringPoolRef,
    @unit_size_t_group
    val name: ResStringPoolRef
)

data class ResXmlTreeEndElement(
    @unit_size_t_group
    val node: ResXmlTreeNode,
    @unit_size_t_group
    val ext: ResXmlTreeEndElementExt
) : IResXmlTreeNode
// endregion

// region : ResXmlTreeStartNamespace
@ResStringPoolRefContainer
data class ResXmlTreeNamespaceExt(
    @unit_size_t_group
    val prefix: ResStringPoolRef,
    @unit_size_t_group
    val uri: ResStringPoolRef
)

data class ResXmlTreeStartNamespace(
    @unit_size_t_group
    val node: ResXmlTreeNode,
    @unit_size_t_group
    val ext: ResXmlTreeNamespaceExt
) : IResXmlTreeNode {

    val prefix by lazy { ext.prefix.value!! }

    val uri by lazy { ext.uri.value!! }
}
// endregion

// region : ResXmlTreeEndNamespace
data class ResXmlTreeEndNamespace(
    @unit_size_t_group
    val node: ResXmlTreeNode,
    @unit_size_t_group
    val ext: ResXmlTreeNamespaceExt
) : IResXmlTreeNode {

    val prefix by lazy { ext.prefix.value!! }

    val uri by lazy { ext.uri.value!! }
}
// endregion

abstract class AbsResXmlTreeNodePair<S : IResXmlTreeNode, E : IResXmlTreeNode> :
    IResStringPoolRefContainer {

    private lateinit var _start: S
    val start: S
        get() = _start

    fun setStart(start: S) {
        _start = start
    }

    private lateinit var _end: E

    val end: E
        get() = _end

    fun setEnd(end: E) {
        _end = end
    }

    override fun updateWithResStringPool(resStringPool: ResStringPool) {
        _start.updateWithResStringPool(resStringPool)
        _end.updateWithResStringPool(resStringPool)
    }
}

class ResXmlTreeNodeNamespace :
    AbsResXmlTreeNodePair<ResXmlTreeStartNamespace, ResXmlTreeEndNamespace>()

class ResXmlTreeNodeElement :
    AbsResXmlTreeNodePair<ResXmlTreeStartElement, ResXmlTreeEndElement>() {

    val name by lazy { start.name }

    val attributes by lazy { start.attributes }

    private var _parent: ResXmlTreeNodeElement? = null

    val parent: ResXmlTreeNodeElement?
        get() = _parent

    fun setParent(parent: ResXmlTreeNodeElement) {
        if (_parent != null) {
            throw IllegalStateException("Cannot set parent twice.")
        }
        _parent = parent
        parent._children.add(this)
    }

    private val _children = mutableListOf<ResXmlTreeNodeElement>()

    val children: List<ResXmlTreeNodeElement>?
        get() = _children.takeIf { it.isNotEmpty() }?.toImmutableList()

    override fun updateWithResStringPool(resStringPool: ResStringPool) {
        super.updateWithResStringPool(resStringPool)
        _children.forEach { child ->
            child.updateWithResStringPool(resStringPool)
        }
    }
}

fun castAsTreeNode(byteList: ByteList): IResXmlTreeNode {
    val header = cast<ResChunkHeader>(byteList)
    return when (header.type) {
        // RES_XML_FIRST_CHUNK_TYPE is same as RES_XML_START_NAMESPACE_TYPE
        ResXmlType.RES_XML_FIRST_CHUNK_TYPE,
        ResXmlType.RES_XML_START_NAMESPACE_TYPE -> cast<ResXmlTreeStartNamespace>(byteList)

        ResXmlType.RES_XML_END_NAMESPACE_TYPE -> cast<ResXmlTreeEndNamespace>(byteList)
        ResXmlType.RES_XML_START_ELEMENT_TYPE -> cast<ResXmlTreeStartElement>(byteList)
        ResXmlType.RES_XML_END_ELEMENT_TYPE -> cast<ResXmlTreeEndElement>(byteList)
        else -> throw IllegalStateException("type ${header.type} is not ResXmlTreeNode.")
    }
}