/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc

import java.nio.ByteBuffer

/**
 * @author Tao Cheng (tao@paradisehell.org)
 */

data class ByteList(private val data: List<Byte>) {

    val size: Int = data.size

    private var _currentIndex = 0

    val currentIndex: Int
        get() {
            return _currentIndex
        }

    val isValid: Boolean
        get() {
            return currentIndex < size
        }

    operator fun plusAssign(move: Int) {
        _currentIndex += move
    }

    operator fun minusAssign(move: Int) {
        _currentIndex -= move
    }


    fun castToInt(fromIndexOffset: Int, toIndexOffset: Int): Int {
        return subList(fromIndexOffset, toIndexOffset).toInt()
    }

    fun castToIntAndPlusAssign(size: Int): Int {
        val value = castToInt(0, size)
        plusAssign(size)
        return value
    }

    fun subList(fromIndexOffset: Int): List<Byte> {
        return subList(fromIndexOffset, data.size - currentIndex)
    }

    fun subList(fromIndexOffset: Int, toIndexOffset: Int): List<Byte> {
        return subList(currentIndex, fromIndexOffset, toIndexOffset)
    }

    fun subList(baseIndex: Int, fromIndexOffset: Int, toIndexOffset: Int): List<Byte> {
        val fromIndex = baseIndex + fromIndexOffset
        val toIndex = baseIndex + toIndexOffset
        return data.subList(fromIndex, toIndex)
    }
}

fun List<Byte>.toLong(): Long {
    return when (size) {
        1 -> {
            val b0 = this[0].toLong()
            b0 and 0xff
        }

        2 -> {
            val b0 = this[0].toLong()
            val b1 = this[1].toLong()
            ((b1 shl 8) and 0xff_ff) or (b0 and 0xff)
        }

        3 -> {
            val b0 = this[0].toLong()
            val b1 = this[1].toLong()
            val b2 = this[2].toLong()
            return ((b2 shl 16) and 0xff_ff_ff) or ((b1 shl 8) and 0xff_ff) or (b0 and 0xff)
        }

        4 -> {
            val b0 = this[0].toLong()
            val b1 = this[1].toLong()
            val b2 = this[2].toLong()
            val b3 = this[3].toLong()
            return ((b3 shl 24) and 0xff_ff_ff_ff) or ((b2 shl 16) and 0xff_ff_ff) or ((b1 shl 8) and 0xff_ff) or (b0 and 0xff)
        }

        else -> throw IllegalStateException("Invalid size, size must be 1, 2 , 3 or 4")
    }
}

fun List<Byte>.toInt(): Int {
    return toLong().toInt()
}

fun List<Byte>.toHex(): String {
    val hex = map { listOf(it) }.reversed().joinToString("") {
        val s = Integer.toHexString(it.toInt())
        if (s.length == 1) "0$s" else s
    }
    return "0x$hex"
}

fun Int.toByteList(size: Int): List<Byte> {
    val byteList = ByteBuffer.allocate(4).putInt(this).array().toList()
    return when (size) {
        1 -> listOf(byteList[3])
        2 -> listOf(byteList[3], byteList[2])
        3 -> listOf(byteList[3], byteList[2], byteList[1])
        4 -> listOf(byteList[3], byteList[2], byteList[1], byteList[0])
        else -> throw IllegalStateException("Invalid size, size must be 1, 2, 3 or 4")
    }
}

fun List<Byte>.toByteList() = ByteList(this)