/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc.single

import com.bytedance.ultimate.inflater.plugin.arsc.*
import java.lang.reflect.Field

/**
 * Created by chentao.joe on 2022/8/14
 * @author chentao.joe@bytedance.com
 */
class ResStringPoolGenerator(private val resStringPool: ResStringPool) : IResXmlTreeChunkGenerator {

    override fun generate(chain: IResXmlTreeChunkGenerator.Chain): List<Byte> {
        val resStringPoolIndexMap = chain.intermediate.resStringPoolIndexMap
        //==================== 1. handle string ====================//
        // 1.1 entries
        val stringByteList = resStringPool.strings
            .mapIndexed { index, stringInfo -> index to stringInfo }
            .filter { resStringPoolIndexMap.contains(it.first) }
            .mapIndexed { index, pair ->
                pair.second.data.let { bytes ->
                    if (index == resStringPoolIndexMap.size - 1) bytes else bytes.plus(0)
                }
            }
        val entryIndexList = mutableListOf<Int>()
        var entryIndex = 0
        stringByteList.forEach { bytes ->
            entryIndexList.add(entryIndex)
            entryIndex += bytes.size
        }
        val entries = entryIndexList.map { it.toByteList(4) }.flatten()
        // 1.2 strings
        val strings = stringByteList.flatten().toMutableList()
        var stringsAlignCount = 4 - (strings.size % 4)
        if (stringsAlignCount == 4) {
            stringsAlignCount = 0
        }
        if (stringsAlignCount == 0 && strings.last() != 0.toByte()) {
            stringsAlignCount = 4
        }
        for (i in 0 until stringsAlignCount) {
            strings.add(0.toByte())
        }

        //==================== 2. handle style ====================//
        // 2.1 entryStyles
        val styleInfoList = resStringPoolIndexMap.originalIndexes
            .mapNotNull { originalIndex -> resStringPool.styles.getOrNull(originalIndex) }
        val stylesByteList = styleInfoList.map { it.data }
        val entryStylesIndexList = mutableListOf<Int>()
        var entryStylesIndex = 0
        stylesByteList.forEach {
            entryStylesIndexList.add(entryStylesIndex)
            entryStylesIndex += it.size
        }
        val entryStyles = entryStylesIndexList.map { it.toByteList(4) }.flatten()
        // 2.2 styles
        val styles = stylesByteList.flatten().toMutableList()
        if (styleInfoList.size > 1) {
            val lastStyle = styleInfoList.last()
            val spanSize = sizeOf(ResStringPoolSpan::class)
            val stylesAlignCount = spanSize - (lastStyle.data.size % spanSize)
            for (i in 0 until stylesAlignCount) {
                styles.add(0.toByte())
            }
        }

        //==================== 3. handle header ====================//
        // 3.2 stringCount
        val stringCount = entryIndexList.size
        // 3.1 stringsStart
        val stringsStart = sizeOf(ResStringPoolHeader::class.java) + entries.size + entryStyles.size
        // 3.3 styleCount
        val styleCount = entryStylesIndexList.size
        // 3.4 stylesStart
        val stylesStart = if (styleCount == 0) 0 else (stringsStart + strings.size)
        // 3.5 size
        val size = if (styleCount == 0) (stringsStart + strings.size) else (stylesStart + styles.size)
        return resStringPool
            .unCast(object : ResXmlTreeFieldUnCastInterceptor() {
                override fun interceptResXmlChunkHeaderSize(field: Field, value: Any): List<Byte> {
                    return size.toByteList(sizeOf(field))
                }

                override fun interceptResStringPoolHeaderStringCount(
                    field: Field,
                    value: Any
                ): List<Byte> {
                    return stringCount.toByteList(sizeOf(field))
                }

                override fun interceptResStringPoolHeaderStyleCount(field: Field, value: Any): List<Byte> {
                    return styleCount.toByteList(sizeOf(field))
                }

                override fun interceptResStringPoolHeaderStringsStart(
                    field: Field,
                    value: Any
                ): List<Byte> {
                    return stringsStart.toByteList(sizeOf(field))
                }

                override fun interceptResStringPoolHeaderStylesStart(field: Field, value: Any): List<Byte> {
                    return stylesStart.toByteList(sizeOf(field))
                }

                override fun interceptResStringPoolEntries(field: Field, value: Any): List<Byte> {
                    return entries
                }

                override fun interceptResStringPoolEntryStyles(field: Field, value: Any): List<Byte> {
                    return entryStyles
                }

                override fun interceptResStringPoolStyles(field: Field, value: Any): List<Byte> {
                    return styles
                }

                override fun interceptResStringPoolStrings(field: Field, value: Any): List<Byte> {
                    return strings
                }
            })
            .plus(chain.process())
    }
}