/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc.single

import com.bytedance.ultimate.inflater.plugin.arsc.ResStringPoolRef
import com.bytedance.ultimate.inflater.plugin.arsc.ResXmlTreeNodeElement
import com.bytedance.ultimate.inflater.plugin.arsc.ResXmlTreeNodeNamespace
import com.bytedance.ultimate.inflater.plugin.arsc.getAllResStringPoolRefs

/**
 * Created by chentao.joe on 2022/8/14
 * @author chentao.joe@bytedance.com
 */
class ResStringPoolIndexMap(
    element: ResXmlTreeNodeElement,
    namespaces: List<ResXmlTreeNodeNamespace>
) {

    private val stringPoolIndexMap = mutableMapOf<Int, Int>()

    val size: Int
        get() = stringPoolIndexMap.size

    val originalIndexes: List<Int>
        get() = stringPoolIndexMap.keys.toList().sorted()

    init {
        // 1. get all ResStringPoolRef from ResXmlTreeNodeElement
        val stringPoolRefs =
            listOfNotNull(element.start.getAllResStringPoolRefs(), element.end.getAllResStringPoolRefs())
                .flatten()
                .filter { it.index >= 0 }
                .associateBy { it.index }
                .toMutableMap()
        // 2. find all ResStringPoolRef from ResXmlTreeNodeNamespace
        namespaces.forEach { namespace ->
            listOfNotNull(
                namespace.start.getAllResStringPoolRefs(),
                namespace.end.getAllResStringPoolRefs()
            ).flatten().filter { it.index > 0 }.let { namespaceRefs ->
                if (namespaceRefs.any { stringPoolRefs.containsKey(it.index) }) {
                    stringPoolRefs.putAll(namespaceRefs.associateBy { it.index })
                }
            }
        }
        // 3. sort ResStringPoolRef
        stringPoolRefs.values.sortedBy { it.index }.forEachIndexed { index, ref ->
            stringPoolIndexMap[ref.index] = index
        }
    }

    operator fun get(originalIndex: Int): Int {
        if (originalIndex < 0) {
            return originalIndex
        }
        return stringPoolIndexMap[originalIndex] ?: throw IllegalStateException("Invalid index : $originalIndex.")
    }

    operator fun get(originRef: ResStringPoolRef): Int {
        return get(originRef.index)
    }

    operator fun contains(originalIndex: Int): Boolean {
        if (originalIndex < 0) {
            return false
        }
        return stringPoolIndexMap.contains(originalIndex)
    }

    operator fun contains(originalRef: ResStringPoolRef): Boolean {
        return contains(originalRef.index)
    }
}