/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.arsc.single

import com.android.build.gradle.internal.utils.toImmutableList
import com.bytedance.ultimate.inflater.plugin.arsc.*

/**
 * Created by chentao.joe on 2022/8/14
 * @author chentao.joe@bytedance.com
 */
@Suppress("MemberVisibilityCanBePrivate")
class SingleNodeResXmlTree {
    private lateinit var _layoutFullName: String
    val layoutFullName: String
        get() = _layoutFullName

    private lateinit var _viewName: String
    val viewName: String
        get() = _viewName


    private lateinit var _data: List<Byte>
    val data: List<Byte>
        get() = _data

    private lateinit var _attributes: Map<String, Any>

    val attributes: Map<String, Any>
        get() = _attributes

    @Suppress("UNCHECKED_CAST")
    fun <T> getAttribute(name: String): T? {
        return attributes[name] as T
    }

    fun <T> requireAttribute(name: String): T {
        return getAttribute<T>(name)
            ?: throw IllegalStateException("Have no attribute with name : $name.")
    }

    private val _children = mutableListOf<SingleNodeResXmlTree>()

    val children: List<SingleNodeResXmlTree>?
        get() = _children.takeIf { it.isNotEmpty() }?.toImmutableList()

    companion object {
        fun parse(tree: ResXmlTree, layoutFullName: String): SingleNodeResXmlTree {
            return createSingleNodeResXmlTree(
                layoutFullName,
                tree.header,
                tree.stringPool,
                tree.resourceMap,
                tree.namespaces,
                tree.rootNode
            )
        }

        private fun createSingleNodeResXmlTree(
            layoutFullName: String,
            header: ResXmlTreeHeader,
            stringPool: ResStringPool,
            resourceMap: ResXmlResourceMap,
            namespaces: List<ResXmlTreeNodeNamespace>,
            element: ResXmlTreeNodeElement
        ): SingleNodeResXmlTree {
            val nodeTree = ResXmlTreeGenerator.elementAsResXmlTree(
                header,
                stringPool,
                resourceMap,
                namespaces,
                element
            ).toResXmlTree()
            return SingleNodeResXmlTree().apply {
                _layoutFullName = layoutFullName
                _data = nodeTree.dataList
                _viewName = nodeTree.rootNode.name
                _attributes = nodeTree.rootNode.attributes
                element.children?.map { childElement ->
                    createSingleNodeResXmlTree(
                        layoutFullName,
                        header,
                        stringPool,
                        resourceMap,
                        namespaces,
                        childElement
                    )
                }?.let { allChildNodes ->
                    _children.addAll(allChildNodes)
                }
            }
        }
    }
}